<?php

namespace App\Models\Concerns;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;

/**
 * Trait Archive.
 *
 *
 * @method Builder archived( int $archivedStatus = 0 ) @see Archive::scopeArchived()
 */
trait Archive
{
    /**
     * Local Scope for retrieving resources by archived status.
     */
    public function scopeArchived(Builder $builder, int $archivedStatus = 0): Builder
    {
        $column = $this->getTable().'.archived_at';
        // archived = 0 : only not archived
        if ($archivedStatus == 0) {
            $builder->whereNull($column);
        } elseif ($archivedStatus == 1) {
            // archived = 1 : only archived
            $builder->whereNotNull($column);
        }

        // else archived = 2 : include both archived and non archived

        return $builder;
    }

    /**
     * Archive the model.
     *
     * @param  callable|null  $callback fired after archived
     * @return bool <false> if already archived
     */
    public function archive(?callable $callback = null): bool
    {
        // is already archived
        if (! empty($this->archived_at)) {
            return false;
        }

        $this->archived_at = Carbon::now();

        $saved = $this->save();

        if ($saved) {
            $this->fireModelEvent('archived', false);

            if ($callback) {
                $callback();
            }
        }

        return $saved;
    }

    /**
     * Unarchived the model.
     *
     * @param  callable|null  $callback fired after unarchived
     */
    public function unarchived(?callable $callback = null): bool
    {
        // is already unarchived
        if (empty($this->archived_at)) {
            return false;
        }

        $this->archived_at = null;

        $saved = $this->save();

        if ($saved) {
            $this->fireModelEvent('unarchived', false);

            if ($callback) {
                $callback();
            }
        }

        return $saved;
    }

    /**
     * Determined resource is archived.
     */
    public function isArchived(): bool
    {
        return ! is_null($this->archived_at);
    }

    /**
     * Get the observable event names.
     *
     * add archived events
     */
    public function getObservableEvents(): array
    {
        return array_merge(
            [
                'archived',
                'unarchived',
            ],
            parent::getObservableEvents()
        );
    }
}
