<?php

namespace App\Models;

use App\Models\Concerns\BulkImport;
use App\Models\Concerns\LogsActivity;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Kirschbaum\PowerJoins\PowerJoins;
use Spatie\Activitylog\LogOptions;

/**
 * Class BackorderQueueCoverage.
 *
 * @property-read int $id
 * @property int $backorder_queue_id
 * @property int $purchase_order_line_id
 * @property int $covered_quantity
 * @property int $released_quantity
 * @property int $unreleased_quantity
 * @property bool $is_tight
 * @property Carbon $created_at
 * @property Carbon|null $updated_at
 * @property-read PurchaseOrderLine $purchaseOrderLine
 * @property-read BackorderQueue $backorderQueue
 *
 * @method Builder|BackorderQueue active()
 */
class BackorderQueueCoverage extends Model
{
    use HasFactory;
    use LogsActivity;
    use PowerJoins;

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    protected $fillable = [
        'backorder_queue_id',
        'covered_quantity',
        'released_quantity',
        'is_tight',
        'purchase_order_line_id',
    ];

    protected $casts = [
        'is_tight' => 'bool',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->backorderQueue->salesOrderLine->salesOrder->id;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrderLine(): BelongsTo
    {
        return $this->belongsTo(PurchaseOrderLine::class);
    }

    public function backorderQueue(): BelongsTo
    {
        return $this->belongsTo(BackorderQueue::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeActive(Builder $builder)
    {
        return $builder->where('unreleased_quantity', '>', '0');
    }
}
