<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporters\BackOrderQueueImporter;
use App\Jobs\ShipStation\AutoFulfillmentOrder;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\BackorderQueueRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\DB;
use Kirschbaum\PowerJoins\PowerJoins;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class BackorderQueue.
 *
 * @property-read int $id
 * @property int $sales_order_line_id
 * @property Carbon $backorder_date
 * @property int|null $supplier_id
 * @property int $priority
 * @property int $backordered_quantity
 * @property int $released_quantity
 * @property int $shortage_quantity
 * @property bool $is_fully_released
 * @property Carbon|null $scheduled_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read SalesOrderLine $salesOrderLine
 * @property-read BackorderQueueRelease[] $backorderQueueReleases
 * @property-read BackorderQueueCoverage[]|Collection $backorderQueueCoverages
 * @property-read Supplier|null $supplier
 * @property-read int $unreleased_quantity
 * @property-read int $covered_quantity
 * @property-read int $unallocated_backorder_quantity
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 *
 * @method Builder|BackorderQueue active( bool $active = true )
 * @method Builder|BackorderQueue releasedByPurchaseOrder( string $purchaseOrderNumber = null )
 */
class BackorderQueue extends Model implements Filterable, HasReference, MapsExportableFields, Sortable
{
    use Archive,
        LogsActivity,
        BulkImport,
        HasFactory,
        HasFilters,
        HasSort,
        PowerJoins;

    const REORDER_TYPE_MOVE_TO_TOP = 'move_to_top';

    const REORDER_TYPE_MOVE_TO_BOTTOM = 'move_to_bottom';

    const REORDER_TYPE_MOVE_UP = 'move_up';

    const REORDER_TYPE_MOVE_DOWN = 'move_down';

    const REORDER_TYPE_MOVE_ABOVE_ID = 'move_above_id';

    const REORDER_TYPE_BULK_MOVE_TO_TOP = 'bulk_move_to_top';

    const REORDER_TYPE_BULK_MOVE_TO_BOTTOM = 'bulk_move_to_bottom';

    const REORDER_TYPE_BULK_MOVE_ABOVE_ID = 'bulk_move_above_id';

    const REORDER_TYPES = [
        self::REORDER_TYPE_MOVE_TO_TOP,
        self::REORDER_TYPE_MOVE_TO_BOTTOM,
        self::REORDER_TYPE_MOVE_UP,
        self::REORDER_TYPE_MOVE_DOWN,
        self::REORDER_TYPE_MOVE_ABOVE_ID,
        self::REORDER_TYPE_BULK_MOVE_TO_TOP,
        self::REORDER_TYPE_BULK_MOVE_TO_BOTTOM,
        self::REORDER_TYPE_BULK_MOVE_ABOVE_ID,
    ];

    protected $guarded = [];

    protected $casts = [
        'priority' => 'int',
        'backorder_date' => 'datetime',
    ];

    public function salesOrderLine()
    {
        return $this->belongsTo(SalesOrderLine::class);
    }

    public function backorderQueueReleases(): HasMany
    {
        return $this->hasMany(BackorderQueueRelease::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function getReference(): ?string
    {
        return $this->id;
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->salesOrderLine->salesOrder->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sales_order_line_id' => $this->sales_order_line_id,
            'backordered_quantity' => $this->backordered_quantity,
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }


    public function getUnreleasedQuantityAttribute()
    {
        return max($this->backordered_quantity - $this->released_quantity, 0);
    }

    public function getSkuAttribute()
    {
        return $this->salesOrderLine && $this->salesOrderLine->product ? $this->salesOrderLine->product->sku : null;
    }

    public function getCoveredByPOAttribute(): string
    {
        $purchaseOrderNumbers = '';
        foreach ($this->backorderQueueCoverages as $coverage) {
            $amountOfBackOrderCoverages = $this->backorderQueueCoverages->count();
            $purchase = $coverage->purchaseOrderLine->purchaseOrder;
            $purchaseOrderNumbers .= $purchase->purchase_order_number.' (covered quantity: '.(($coverage->covered_quantity - $coverage->released_quantity) ?? 0).')'.($amountOfBackOrderCoverages > 1 ? ', ' : '');
        }

        return $purchaseOrderNumbers;
    }

    public function save(array $options = [])
    {
        $this->salesOrderLine->setBackorderStatus(! $this->is_fully_released);

        if (max(0, $this->backordered_quantity - $this->released_quantity) == 0) {
            $this->priority = null;
        }

        return parent::save($options);
    }

    public function setReleasedQuantityAttribute($value)
    {
        // We reset the priority when the queue comes back on as active.
        $this->attributes['released_quantity'] = $value;
    }

    public function setPriority(int $priority)
    {
        $this->priority = $priority;
        $this->save();
    }

    public function getProductId(): int
    {
        return $this->salesOrderLine->product_id;
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'layer');
    }

    public function backorderQueueCoverages(): HasMany
    {
        return $this->hasMany(BackorderQueueCoverage::class);
    }

    public function getReleasedQuantityAttribute()
    {
        // Not sure why this logic exists?
        //        if (is_null($this->priority)) {
        //            return $this->attributes['backordered_quantity'];
        //        } else {
        //            return isset($this->attributes['released_quantity']) ? $this->attributes['released_quantity'] : 0;
        //        }

        return isset($this->attributes['released_quantity']) ? $this->attributes['released_quantity'] : 0;
    }

    /**
     * @return int|mixed
     */
    public function getUnallocatedBackorderQuantityAttribute()
    {
        // Quantity without coverages
        return max(0, $this->unreleased_quantity - $this->covered_quantity);
    }

    public function getCoveredQuantityAttribute()
    {
        return $this->backorderQueueCoverages->sum('unreleased_quantity');
    }

    /**
     * @throws Exception|Throwable
     */
    public function addRelease(
        ?int $linkId,
        string $linkType,
        int $quantity,
        FifoLayer $fifoLayer,
        bool $suppressCoveragesSync = false
    ): ?BackorderQueueRelease {
        if (! $quantity || (! $this->salesOrderLine->salesOrder->isOpen() && ! $this->salesOrderLine->salesOrder->isReserved())) {
            return null;
        }

        /** @var BackorderQueueRelease $release */
        $release = $this->backorderQueueReleases()
            ->create([
                'link_type' => $linkType,
                'link_id' => $linkId,
                'released_quantity' => $quantity,
            ]);

        $this->released_quantity = min($this->backordered_quantity, $this->released_quantity + $quantity);
        if ($this->is_fully_released) {
            // Decrement all priorities after this queue
            /** @var BackorderQueueRepository $queues */
            $queues = app()->make(BackorderQueueRepository::class);
            $queues->decrementPrioritiesAfter(
                $this->priority,
                $this->salesOrderLine->product_id,
                1,
                [$this->id]
            );

            // Set this queue's priority to null since it's fully released.
            $this->priority = null;
            $this->released_quantity = $this->backordered_quantity;
        }

        $this->save();

        return $this->afterRelease($release, $fifoLayer, $suppressCoveragesSync);
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     */
    protected function afterRelease(BackorderQueueRelease $release, FifoLayer $fifoLayer, bool $suppressCoveragesSync = false): BackorderQueueRelease
    {
        if ($this->salesOrderLine->product_id && $this->salesOrderLine->warehouse_id && !$suppressCoveragesSync) {
            // Next, we sync the backorder queue coverages for the product.
            dispatch(new SyncBackorderQueueCoveragesJob(null, null, [$this->salesOrderLine->product_id], $this->salesOrderLine->warehouse_id));
        }

        // Release movements
        if ($this->unreleased_quantity) {
            $this->transferMovementsToFifo($this->inventoryMovements, $fifoLayer, $release->released_quantity);
        } else {
            // The backorder queue is fully released,
            // we set the movements to the fifo layer
            // TODO:: Using each instead of bulk update
            // TODO:: like below just to test model auditing
            // TODO:: since bulk update doesn't fire model events.
            $this->inventoryMovements->each->update([
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
            ]);
            //            $this->inventoryMovements()->update([
            //                'layer_id'   => $fifoLayer->id,
            //                'layer_type' => FifoLayer::class,
            //            ]);
        }
        // add a link between sales order line and fifo layer
        $this->salesOrderLine->fifoLayers()->attach($fifoLayer->id, ['quantity' => $release->released_quantity]);

        /**
         * Charge the fifo layer.
         */
        $fifoLayer->fulfilled_quantity += $release->released_quantity;
        $fifoLayer->save();

        /**
         * Auto fulfill the sales order if the warehouses of lines belong to the automated warehouses.
         * Note that we run auto fulfillment script in updateSKUOrder method.
         */
        ( new AutoFulfillmentOrder($this->salesOrderLine->salesOrder) )->fulfillOpenOrder();

        return $release;
    }

    /**
     * @throws Throwable
     */
    public static function syncCoveragesForProducts(int|array $products, ?int $warehouseId = null): void
    {
        DB::beginTransaction();
        try {
            $products = array_unique((array) $products);

            // We re-calculate the backorder queue coverages
            // based on PO eta and order_date as fallback.
            BackorderQueueCoverage::query()
                ->joinRelationship('purchaseOrderLine.purchaseOrder')
                ->whereIn('purchase_order_lines.product_id', $products)
                ->when($warehouseId, function ($query) use ($warehouseId) {
                    $query->where('purchase_orders.destination_warehouse_id', $warehouseId);
                })->delete();

            /** @var BackorderQueueRepository $queues */
            $queues = app(BackorderQueueRepository::class);

            PurchaseOrderLine::with(['purchaseOrderShipmentReceiptLines'])
                ->selectRaw('purchase_order_lines.*, purchase_orders.estimated_delivery_date, purchase_orders.purchase_order_date')
                ->joinRelationship('purchaseOrder')
                ->leftJoinRelationship('purchaseOrderShipmentReceiptLines')
                ->whereIn('purchase_order_lines.product_id', $products)
                ->when($warehouseId, function ($query) use ($warehouseId) {
                    $query->where('purchase_orders.destination_warehouse_id', $warehouseId)
                        ->where('receipt_status', '!=', PurchaseOrder::RECEIPT_STATUS_DROPSHIP);
                })
                ->where('purchase_orders.order_status', PurchaseOrder::STATUS_OPEN)
                ->orderByRaw('purchase_orders.estimated_delivery_date, purchase_orders.purchase_order_date')
                ->each(function (PurchaseOrderLine $orderLine) use ($queues) {
                    $queues->purchaseOrderCoverage($orderLine->unreceived_quantity, $orderLine);
                });
            DB::commit();
        } catch (Throwable $e) {
            try {
                DB::rollBack();
            } catch (Throwable $rollbackException) {
                //                throw new QueryException($rollbackException->getMessage(), [], $e);
            }
            // We let the developer handle any exceptions that may
            // have occurred with the syncing of the coverages.
            throw $e;
        }
    }

    /**
     * @throws Exception|Throwable
     */
    public function addFBARelease(
        ?int $linkId,
        string $linkType,
        int $quantity,
        FifoLayer $fifoLayer
    ): ?BackorderQueueRelease {
        if (! $quantity || ! $this->salesOrderLine->salesOrder->isOpen()) {
            return null;
        }

        /** @var BackorderQueueRelease $release */
        $release = $this->backorderQueueReleases()
            ->create([
                'link_type' => $linkType,
                'link_id' => $linkId,
                'released_quantity' => $quantity,
            ]);

        $this->released_quantity = min($this->backordered_quantity, $this->released_quantity + $quantity);
        if ($this->is_fully_released) {
            // Decrement all priorities after this queue
            /** @var BackorderQueueRepository $queues */
            $queues = app()->make(BackorderQueueRepository::class);
            $queues->decrementPrioritiesAfter(
                $this->priority,
                $this->salesOrderLine->product_id,
                1,
                [$this->id]
            );

            // Set this queue's priority to null since it's fully released.
            $this->priority = null;
        }

        $this->save();

        return $this->afterRelease($release, $fifoLayer);
    }

    public function getIsFullyReleasedAttribute(): bool
    {
        return $this->backordered_quantity <= $this->released_quantity;
    }

    /**
     * @throws Throwable
     */
    public function delete()
    {
        // Attempt to delete backorder with related records.
        $result = DB::transaction(function () {
            // Delete coverages and releases
            $this->backorderQueueCoverages()->delete();
            $this->backorderQueueReleases()->delete();
            // Delete inventory movements
            $this->inventoryMovements()->delete();

            return parent::delete();
        });

        // No inventory movements should be left behind.
        if ($result && $this->inventoryMovements()->count() > 0) {
            // Attempt to delete any orphan movements.
            $this->inventoryMovements()->each(function (InventoryMovement $movement) {
                $movement->delete();
            });
        }

        return $result;
    }

    protected function transferMovementsToFifo($movements, FifoLayer $fifoLayer, int $quantity)
    {
        foreach ($movements as $movement) {
            $fifoLayerMovement = $movement->replicate();
            // active movement to the sales order line means it's a negative movement
            if ($movement->inventory_status == InventoryMovement::INVENTORY_STATUS_ACTIVE) {
                $movement->quantity += $quantity;
                $fifoLayerMovement->quantity = -abs($quantity);
            } else {
                $movement->quantity -= $quantity;
                $fifoLayerMovement->quantity = abs($quantity);
            }
            // update backorder movement
            $movement->save();
            // create a new movement for the sales order line
            $fifoLayerMovement->fifo_layer = $fifoLayer->id;
            $fifoLayerMovement->save();
        }
    }

    public static function getExportableFields(): array
    {
        return BackOrderQueueImporter::getExportableFields();
    }

    public function availableColumns()
    {
        return config('data_table.backorder_queue.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['sku', 'product_name', 'supplier', 'sales_order'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())
            ->where('sortable', 1)
            ->pluck('data_name')
            ->all();
    }

    public function scopeActive($builder)
    {
        return $builder->where('shortage_quantity', '>', 0);
        //        return $builder->whereNotNull('priority')
        //          ->whereColumn('backordered_quantity', '!=', 'released_quantity');
    }

    public function scopeReleasedByPurchaseOrder(Builder $builder, ?string $purchaseOrderNumber = null)
    {
        return $builder->whereHas('backorderQueueReleases', function (Builder $builder) use ($purchaseOrderNumber) {
            $builder->whereHasMorph('link', PurchaseOrderShipmentReceiptLine::class, function (Builder $builder) use ($purchaseOrderNumber) {
                if (! empty($purchaseOrderNumber)) {
                    $builder->whereHas('purchaseOrderShipmentLine.purchaseOrderLine.purchaseOrder', function (Builder $builder) use ($purchaseOrderNumber) {
                        $builder->where('purchase_order_number', $purchaseOrderNumber);
                    });
                }
            });
        });
    }

    public function scopeCoveredByPurchaseOrder(Builder $builder, string|array|null $purchaseOrderNumber = null, $operator = null)
    {
        return $builder->whereHas('backorderQueueCoverages', function (Builder $builder) use ($purchaseOrderNumber, $operator) {
            if (! empty($purchaseOrderNumber)) {
                $builder->whereHas('purchaseOrderLine.purchaseOrder', function (Builder $builder) use ($purchaseOrderNumber, $operator) {
                    $builder->filterKey(['is_relation' => true, 'key' => 'purchase_order_number'], $operator, $purchaseOrderNumber);
                });
            }
        });
    }

    public function getSchedule(): ?array
    {
        if ($this->supplier && $this->supplier->auto_generate_backorder_po && $this->unreleased_quantity) {
            // There is an unreleased quantity and we auto generate POs for backorders for the
            // supplier, so we add in the next schedule.
            return [
                'quantity' => $this->unreleased_quantity - $this->covered_quantity,
                'supplier' => [
                    'id' => $this->supplier->id,
                    'name' => $this->supplier->name,
                ],
                'scheduled_at' => $this->supplier->getNextBackorderSchedule(),
            ];
        }

        return null;
    }

    public function scopeFilterSalesOrderLine(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 2));
            $lastKey = array_slice($keyExploded, -1)[0];

            if ($relation === 'salesOrderLine.product') {
                if (in_array($lastKey, ['sku', 'name'])) {
                    return $builder->{$function.'Has'}('salesOrderLine.product', function (Builder $builder) use ($lastKey, $operator, $value) {
                        return $builder->filterKey([
                            'key' => $builder->qualifyColumn($lastKey),
                            'is_relation' => false,
                        ], $operator, $value);
                    });
                }

                if (! $this->isTableJoined($builder, 'sales_order_lines')) {
                    $builder->leftJoin('sales_order_lines', 'backorder_queues.sales_order_line_id', 'sales_order_lines.id');
                }
                if (! $this->isTableJoined($builder, 'products')) {
                    $builder->leftJoin('products', 'sales_order_lines.product_id', 'products.id');
                }

                return $builder->$function('products.'.$lastKey, $operator, $value);
            } elseif ($relation === 'salesOrderLine.salesOrder') {
                return $builder->{$function.'Has'}($relation, function (Builder $builder) use ($lastKey, $operator, $value) {
                    return $builder->filterKey([
                        'key' => $builder->qualifyColumn($lastKey),
                        'is_relation' => false,
                    ], $operator, $value);
                });
            }
        }

        return $builder;
    }

    public function scopeFilterBackorderQueueCoverages(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 3));
            $lastKey = array_slice($keyExploded, -1)[0];

            if ($operator === 'isEmpty') {
                return $builder->whereDoesntHave($relation);
            }

            return $builder->{$function.'Has'}($relation, function (Builder $builder) use ($lastKey, $operator, $value) {
                return $builder->filterKey([
                    'key' => $builder->qualifyColumn($lastKey),
                    'is_relation' => false,
                ], $operator, $value);
            });
        }

        return $builder;
    }

    public function scopeSortSalesOrderLine(Builder $builder, array $relation, bool $ascending)
    {
        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 2));
            $lastKey = array_slice($keyExploded, -1)[0];

            if ($relation === 'salesOrderLine.product') {
                if (! $this->isTableJoined($builder, 'sales_order_lines')) {
                    $builder->leftJoin('sales_order_lines', 'backorder_queues.sales_order_line_id', 'sales_order_lines.id');
                }
                if (! $this->isTableJoined($builder, 'products')) {
                    $builder->leftJoin('products', 'sales_order_lines.product_id', 'products.id');
                }

                $builder->orderBy('products.'.$lastKey, $ascending ? 'asc' : 'desc');
            } elseif ($relation === 'salesOrderLine.salesOrder') {
                if (! $this->isTableJoined($builder, 'sales_order_lines')) {
                    $builder->leftJoin('sales_order_lines', 'backorder_queues.sales_order_line_id', 'sales_order_lines.id');
                }
                if (! $this->isTableJoined($builder, 'sales_orders')) {
                    $builder->leftJoin('sales_orders', 'sales_order_lines.sales_order_id', 'sales_orders.id');
                }

                $builder->orderBy('sales_orders.'.$lastKey, $ascending ? 'asc' : 'desc');
            }
        }

        return $builder;
    }

    public function scopeSortBackorderQueueCoverages(Builder $builder, array $relation, bool $ascending)
    {
        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 3));
            $lastKey = array_slice($keyExploded, -1)[0];

            if ($relation === 'backorderQueueCoverages.purchaseOrderLine.purchaseOrder') {
                if (! $this->isTableJoined($builder, 'backorder_queue_coverages')) {
                    $builder->leftJoin('backorder_queue_coverages', 'backorder_queues.id', 'backorder_queue_coverages.backorder_queue_id');
                }
                if (! $this->isTableJoined($builder, 'purchase_order_lines')) {
                    $builder->leftJoin('purchase_order_lines', 'backorder_queue_coverages.purchase_order_line_id', 'purchase_order_lines.id');
                }
                if (! $this->isTableJoined($builder, 'purchase_orders')) {
                    $builder->leftJoin('purchase_orders', 'purchase_order_lines.purchase_order_id', 'purchase_orders.id');
                }

                $builder->orderBy('purchase_orders.'.$lastKey, $ascending ? 'asc' : 'desc');
            }
        }

        return $builder;
    }

    public function scopeSortSupplier(Builder $builder, array $relation, bool $ascending)
    {
        if (! $this->isTableJoined($builder, 'suppliers')) {
            $builder->leftJoin('suppliers', 'backorder_queues.supplier_id', 'suppliers.id');
        }

        $keyExploded = explode('.', $relation['combined_key']);
        $lastKey = array_slice($keyExploded, -1)[0];

        return $builder->orderBy('suppliers.'.$lastKey, $ascending ? 'asc' : 'desc');
    }

    public function scopeInactive(Builder $builder)
    {
        return $builder->whereColumn('backordered_quantity', '=', 'released_quantity')
            ->orWhereNull('priority');
    }
}
