<?php

namespace App\Models;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Str;

/**
 * Class BackOrderLayer.
 *
 *
 * @property int $id
 * @property int $product_id
 * @property float $quantity_backordered
 * @property float $quantity_released
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 * @property-read float $unreleased_quantity
 */
class BackorderLayer extends Model implements Filterable, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;

    protected $fillable = ['product_id', 'quantity_backordered', 'quantity_released', 'sales_order_line_id'];

    protected $casts = [
        'quantity_backordered' => 'integer',
        'quantity_released' => 'integer',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function salesOrderLine()
    {
        return $this->belongsTo(SalesOrderLine::class);
    }

    public function salesOrderLines()
    {
        return $this->morphToMany(SalesOrderLine::class, 'layer', 'sales_order_line_layers')
            ->using(SalesOrderLineLayer::class)
            ->withPivot('quantity')
            ->withTimestamps();
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'layer');
    }

    public function getUnreleasedQuantityAttribute()
    {
        return $this->quantity_backordered - $this->quantity_released;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.backorder_layer.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->filterableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())
            ->where('sortable', 1)
            ->pluck('data_name')
            ->all();
    }

    private function makeLinkTypeFromName($name): string
    {
        return __NAMESPACE__.'\\'.ucfirst(Str::camel($name));
    }

    public function scopeFilterLink(Builder $builder, array $relation, string $operator, string $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';
        $builder->{$function}('link_type', $this->makeLinkTypeFromName($value));

        return $builder;
    }

    public function scopeSortLink(Builder $builder, array $relation, bool $ascending): Builder
    {
        $builder->orderBy('link_type', $ascending ? 'asc' : 'desc');

        return $builder;
    }

    /**
     * @return BackorderLayer|Builder
     */
    public function scopeSortSalesOrderLines(Builder $builder, array $relation, bool $ascending)
    {
        if (! isset($relation['key'])) {
            return $builder;
        }

        if ($relation['key'] === 'salesOrder') {
            return $this->scopeSortSalesOrder($builder, $relation, $ascending);
        }

        return $builder;
    }

    public function scopeSortSalesOrder(Builder $builder, array $relation, bool $ascending): Builder
    {
        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 2));
            $lastKey = array_slice($keyExploded, -1)[0];

            return $builder->whereHas($relation, function (Builder $builder) use ($lastKey, $ascending) {
                $builder->sortKey([
                    'key' => $builder->qualifyColumn($lastKey),
                    'is_relation' => false,
                ], $ascending);
            });
        }

        //
        //    $parts = explode( '.', $relation['combined_key'] );
        //    $field = end( $parts );
        //
        //    if ( ! $this->isTableJoined( $builder, 'sales_order_lines' ) )
        //    {
        //      $builder->leftJoin( 'sales_order_lines', 'backorder_layers.sales_order_line_id', '=', 'sales_order_lines.id' );
        //    }
        //
        //    if ( ! $this->isTableJoined( $builder, 'sales_orders' ) )
        //    {
        //      $builder->leftJoin( 'sales_orders', 'sales_order_lines.sales_order_id', '=', 'sales_orders.id' );
        //    }
        //
        //    return $builder->orderBy( "sales_orders.{$field}", $ascending ? 'asc' : 'desc' );
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'sku' => Exporter::makeExportableField('sku', false, 'SKU'),
            'product_name' => Exporter::makeExportableField('product_name', false),
            'backorder_quantity' => Exporter::makeExportableField('backorder_quantity', false),
            'released_quantity' => Exporter::makeExportableField('released_quantity', false),
            'sales_order_date' => Exporter::makeExportableField('sales_order_date', false),
            'origin.link_id' => Exporter::makeExportableField('origin_link_id', false),
            'origin.reference' => Exporter::makeExportableField('origin_link_reference', false),
        ];
    }
}
