<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\AttributeGroupDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class AttributeGroup.
 *
 *
 * @property int $id
 * @property string $name
 * @property int|null $sort_order
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class AttributeGroup extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    protected $fillable = ['name', 'sort_order'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function products()
    {
        return $this->hasManyThrough(Product::class, ProductAttribute::class, 'product_id', 'id');
    }

    public function categories()
    {
        return $this->belongsToMany(ProductCategory::class, 'categories_to_attribute_groups', null, 'category_id')
            ->withTimestamps();
    }

    public function categoriesToAttributeGroups()
    {
        return $this->hasMany(CategoryToAttributeGroup::class);
    }

    public function attributes()
    {
        return $this->hasMany(Attribute::class);
    }

    public function productsIdsThroughCategories()
    {
        return $this->hasManyThrough(ProductToCategory::class, CategoryToAttributeGroup::class, 'attribute_group_id', 'category_id', 'id', 'category_id')->selectRaw('DISTINCT(`product_to_categories`.`product_id`)');
    }

    public function productsToAttributeGroups()
    {
        return $this->hasMany(ProductToAttributeGroup::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getProductsCountThroughCategoriesAttribute()
    {
        return count($this->productsIdsThroughCategories);
    }

    /*
    |--------------------------------------------------------------------------
    | Custom Function
    |--------------------------------------------------------------------------
    */

    public function delete()
    {
        // check the attribute group associations
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        // delete its associations
        $this->categories()->detach();
        $this->products()->delete();

        return parent::delete();
    }

    /**
     * Is attribute group has associations.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        if (! $this->relationLoaded('productsIdsThroughCategories')) {
            $this->load('productsIdsThroughCategories');
        }

        if (! property_exists($this, 'products_count')) {
            $this->loadCount('products');
        }

        $this->loadCount('attributes');

        $usage = [];

        if ($this->productsIdsThroughCategories->isNotEmpty()) {
            $usage['productsThroughCategories'] = trans_choice('messages.currently_used', $this->productsIdsThroughCategories->count(), [
                'resource' => $this->productsIdsThroughCategories->count() == 1 ? 'products through category' : 'products through categories',
                'model' => 'attribute group('.$this->name.')',
            ]);
        }

        if ($this->products_count) {
            $usage['products'] = trans_choice('messages.currently_used', $this->products_count, [
                'resource' => 'direct product',
                'model' => 'attribute group('.$this->name.')',
            ]);
        }

        if ($this->attributes_count) {
            $usage['attributes'] = trans_choice('messages.currently_used', $this->attributes_count, [
                'resource' => 'attribute',
                'model' => 'attribute group('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.attribute_group.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * Calculated columns used to filter by "having".
     */
    public function calculatedColumns(): array
    {
        return ['attributes_count', 'categories_to_attribute_groups_count', 'products_count'];
    }

    public function customColumns()
    {
        return ['products_count' => '(select count(*) from `product_attributes` where `attribute_groups`.`id` = `product_attributes`.`attribute_group_id`) + (select count(*) from `product_to_categories` inner join `categories_to_attribute_groups` on `categories_to_attribute_groups`.`category_id` = `product_to_categories`.`category_id` where `attribute_groups`.`id` = `categories_to_attribute_groups`.`attribute_group_id`)'];
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new AttributeGroupDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return AttributeGroupDataImporter::getExportableFields();
    }
}
