<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Casts\EncryptValue;
use App\Exceptions\AddressLockedException;
use App\Helpers;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Constant\ConstantsCountry;
use App\Models\Constant\ConstantsZipcode;
use App\Models\Contracts\Filterable;
use App\SDKs\Starshipit\Model\DestinationDetails;
use Carbon\Carbon;
use DB;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

/**
 * Class Address.
 *
 *
 * @property int $id
 * @property string|null $company
 * @property string $name
 * @property string|null $email
 * @property string|null $phone
 * @property string|null $fax
 * @property string $address1
 * @property string|null $address2
 * @property string|null $address3
 * @property string $city
 * @property string|null $province
 * @property string|null $province_code
 * @property string|null $zip
 * @property string|null $country
 * @property string $country_code
 * @property string|null $label
 * @property string $hash
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read null|ConstantsCountry $constantCountry
 * @property-read mixed $accountingIntegration
 * @property-read SalesOrder[]|Collection $salesOrdersByShippingAddress
 * @property-read SalesOrder[]|Collection $salesOrdersByBillingAddress
 * @property-write string $company_name
 */
class Address extends Model implements Filterable, UniqueFieldsInterface
{
    use BulkImport;
    use HasFactory;
    use HasFilters;
    use Notifiable;

    protected $fillable = [
        'id',
        'company',
        'company_name',
        'name',
        'email',
        'phone',
        'fax',
        'address1',
        'address2',
        'address3',
        'city',
        'province',
        'province_code',
        'zip',
        'country',
        'country_code',
        'label',
        'accounting_integration_id',
        'accounting_integration_type',
        'updated_at',
    ];

    protected $casts = [
        'name' => EncryptValue::class,
        'email' => EncryptValue::class,
        'phone' => EncryptValue::class,
        'address1' => EncryptValue::class,
        'address2' => EncryptValue::class,
        'address3' => EncryptValue::class,
    ];

    protected $hidden = [
        'created_at',
        'updated_at',
        'pivot',
    ];

    public static function getUniqueFields(): array
    {
        return [
            'name',
            'company',
            'email',
            'address1',
            'address2',
            'address3',
            'city',
            'province',
            'province_code',
            'country',
            'country_code',
            'zip',
            'phone',
            'fax'
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrdersByShippingAddress(): HasMany
    {
        return $this->hasMany(SalesOrder::class, 'shipping_address_id');
    }

    public function salesOrdersByBillingAddress(): HasMany
    {
        return $this->hasMany(SalesOrder::class, 'billing_address_id');
    }

    public function customerByShippingAddress(): HasOne
    {
        return $this->hasOne(Customer::class, 'default_shipping_address_id');
    }

    public function customerByBillingAddress(): HasOne
    {
        return $this->hasOne(Customer::class, 'default_billing_address_id');
    }

    public function addressables(): HasMany
    {
        return $this->hasMany(Addressable::class);
    }

    public function customerByOtherAddresses(): MorphToMany
    {
        return $this->morphedByMany(Customer::class, 'addressable');
    }

    public function constantCountry(): BelongsTo
    {
        return $this->belongsTo(ConstantsCountry::class, 'country_code', 'code');
    }

    public function suppliers(): MorphToMany
    {
        return $this->morphedByMany(Supplier::class, 'addressable');
    }

    public function directSuppliers(): HasMany
    {
        return $this->hasMany(Supplier::class);
    }

    public function warehouses(): HasMany
    {
        return $this->hasMany(Warehouse::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function setCompanyNameAttribute($value)
    {
        $this->company = $value;
    }

    public function setPhoneAttribute($value)
    {
        $this->attributes['phone'] = $value ? Helpers::getPhoneAttribute($value, $this->constantCountry) : null;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function save(array $options = []): bool
    {
        if ($this->isLocked()) {
            throw new AddressLockedException('Address is locked and cannot be modified.');
        }

        // Fill in country name using country code
        $this->country = $this->findCountryByCode();

        // Fill in province using province code.
        $this->province = $this->findProvinceByCode();

        // To prevent duplicate addresses, we try to
        // match an existing address with the same hash.
        // Encapsulated in a transaction to avoid race condition (See SKU-6499)
        return DB::transaction(function () use ($options) {
            if ($existingAddress = self::query()->where('hash', $this->getHash())->lockForUpdate()->first()){
                $this->fill($existingAddress->toArray());
                return true;
            } else {
                return parent::save($options);
            }
        });
    }

    /**
     * Computes the unique hash of the address.
     *
     * @return string
     */
    public function getHash(): string{
        return sha1(implode('|', array_values($this->only(static::getUniqueFields()))));
    }

    public function findCountryByCode(): ?string
    {
        if ($this->country) {
            return $this->country;
        }
        if ($this->country_code) {
            /** @var ConstantsCountry $country */
            $country = ConstantsCountry::query()->where('code', $this->country_code)->first();

            if ($country) {
                return $country->name;
            }
        }

        return null;
    }

    public function findProvinceByCode(): ?string
    {
        if ($this->province) {
            return $this->province;
        }
        if (! $this->province_code) {
            return null;
        }

        /**
         * For US and CA, we fetch the province based on code,
         * and for other countries, province is set and code is null (SKU-2446)
         *
         * @var ConstantsZipcode $province
         */
        $province = ConstantsZipcode::with([])->where('state', $this->province_code)
            ->where('country_code', $this->country_code)
            ->first();

        if (! $this->isUSOrCanada()) {
            $this->province_code = null;
        }

        return $province?->full_state;
    }

    public function isUSOrCanada(): bool
    {
        $codes = ['US', 'CA'];
        if ($this->country_code) {
            return in_array($this->country_code, $codes);
        }

        // Try to find by province code
        if ($this->province_code) {
            /** @var ConstantsZipcode $province */
            $province = ConstantsZipcode::with([])->where('state', $this->province_code)->first();

            if ($province) {
                return in_array($province->country_code, $codes);
            }
        }

        return false;
    }

    public function delete(): bool|array|null
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    public function formatForEmail(): string
    {
        $address = "$this->name<br>";
        $address .= $this->address1 ? "$this->address1<br>" : '';
        $address .= $this->address2 ? "$this->address2<br>" : '';
        $address .= $this->address3 ? "$this->address3<br>" : '';
        $address .= $this->city ? ($this->city.($this->province_code || $this->zip ? ', ' : '')) : '';
        $address .= $this->province_code || $this->zip ? "$this->province_code $this->zip<br>" : '<br>';
        $address .= $this->country ?: '';

        return $address;
    }

    public function formatForInvoice(bool $withName = true): string
    {
        $address = $withName ? "$this->name \n" : '';
        $address .= $this->address1." \n";
        $address .= $this->address2 ? $this->address2." \n" : '';
        $address .= $this->address3 ? $this->address3." \n" : '';
        $address .= "$this->city".($this->city ? ', ' : '')."$this->province_code  $this->zip \n";
        $address .= $this->country ? "$this->country \n" : '';
        $address .= $this->phone ? "$this->phone \n" : '';
        $address .= $this->email ? "$this->email" : '';

        return $address;
    }

    /**
     * Determined if the address used by other resources.
     */
    public function isUsed(): bool|array
    {
        $usage = [];

        // Sales Orders
        $shippingCount = $this->salesOrdersByShippingAddress()->count();
        $billingCount = $this->salesOrdersByBillingAddress()->count();
        if ($shippingCount || $billingCount) {
            $usage['salesOrders'] = trans_choice('messages.currently_used', $shippingCount + $billingCount, [
                'resource' => 'sales order',
                'model' => 'address('.$this->name.')',
            ]);
        }

        // Customers
        $shippingCount = $this->customerByShippingAddress()->count();
        $billingCount = $this->customerByBillingAddress()->count();
        if ($shippingCount || $billingCount) {
            $usage['Customer'] = trans_choice('messages.currently_used', $shippingCount + $billingCount, [
                'resource' => 'customer',
                'model' => 'address('.$this->name.')',
            ]);
        }
        //
        // addressables
        $addressableCount = $this->addressables()->count();
        if ($addressableCount) {
            $usage['addressables'] = trans_choice('messages.currently_used', $addressableCount, [
                'resource' => 'resource',
                'model' => 'address('.$this->name.')',
            ]);
        }

        // suppliers
        $suppliersCount = $this->directSuppliers()->count();
        if ($suppliersCount) {
            $usage['suppliers'] = trans_choice('messages.currently_used', $suppliersCount, [
                'resource' => 'supplier',
                'model' => 'address('.$this->name.')',
            ]);
        }

        // warehouses
        $warehousesCount = $this->warehouses()->count();
        if ($warehousesCount) {
            $usage['warehouses'] = trans_choice('There is 1 :resource associated with this :model.|...', $warehousesCount, [
                'resource' => 'warehouse',
                'model' => 'address('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    public function usedBySalesOrders($exceptedSalesOrdersIds = []): bool
    {
        $exceptedSalesOrdersIds = Arr::wrap($exceptedSalesOrdersIds);

        if (empty($exceptedSalesOrdersIds)) {
            $this->loadCount(['salesOrdersByShippingAddress', 'salesOrdersByBillingAddress']);
        } else {
            $this->loadCount([
                'salesOrdersByShippingAddress' => function (Builder $builder) use ($exceptedSalesOrdersIds) {
                    $builder->whereNotIn('id', $exceptedSalesOrdersIds);
                },
                'salesOrdersByBillingAddress' => function (Builder $builder) use ($exceptedSalesOrdersIds) {
                    $builder->whereNotIn('id', $exceptedSalesOrdersIds);
                },
            ]);
        }

        if ($this->sales_orders_by_shipping_address_count || $this->sales_orders_by_billing_address_count) {
            return true;
        }

        return false;
    }

    public function usedByCustomers($exceptCustomersIds = []): bool
    {
        $exceptCustomersIds = Arr::wrap($exceptCustomersIds);

        if (empty($exceptedSalesOrdersIds)) {
            $this->loadCount(['customerByShippingAddress', 'customerByBillingAddress']);
        } else {
            $this->loadCount([
                'customerByShippingAddress' => function (Builder $builder) use ($exceptCustomersIds) {
                    $builder->whereNotIn('id', $exceptCustomersIds);
                },
                'customerByBillingAddress' => function (Builder $builder) use ($exceptCustomersIds) {
                    $builder->whereNotIn('id', $exceptCustomersIds);
                },
            ]);
        }

        if ($this->customer_by_shipping_address_count || $this->customer_by_billing_address_count) {
            return true;
        }

        return false;
    }

    /**
     * Determined if the address is locked(editable, deletable).
     */
    public function isLocked(): bool
    {
        if ($this->isForAmazonCustomer()) {
            return false;
        }
        $this->load('salesOrdersByShippingAddress', 'salesOrdersByBillingAddress');
        $saleOrderStatusPrevented = [SalesOrder::STATUS_OPEN, SalesOrder::STATUS_CLOSED];

        // sales orders by shipping address
        if ($this->salesOrdersByShippingAddress->pluck('order_status')->intersect($saleOrderStatusPrevented)->isNotEmpty()) {
            return true;
        }

        // sales orders by billing address
        if ($this->salesOrdersByBillingAddress->pluck('order_status')->intersect($saleOrderStatusPrevented)->isNotEmpty()) {
            return true;
        }

        return false;
    }

    private function isForAmazonCustomer(): bool
    {
        if ($customer = $this->customerByShippingAddress) {
            return $customer->isAmazon();
        }

        if ($customer = $this->customerByBillingAddress) {
            return $customer->isAmazon();
        }

        return $this->addressables
            ->filter(function (Addressable $addressable) {
                if ($addressable->addressable_type == Customer::class) {
                    /** @var Customer $customer */
                    $customer = $addressable->addressable;
                    return $customer->isAmazon();
                }
                return false;
            })
            ->isNotEmpty();
    }

    /**
     * Check if address exists.
     *
     * We search by "country code" , "zip" , "address1" and "name".
     */
    public function isExists(): Model|self|Builder|null
    {
        return self::query()->where(function (Builder $query) {
            // country_code
            $query->whereNotNull('country_code')
                ->where('country_code', '!=', '')
                ->where('country_code', $this->country_code)
        // zip
                ->whereNotNull('zip')
                ->where('zip', '!=', '')
                ->where('zip', $this->zip)
        // address1
                ->whereNotNull('address1')
                ->where('address1', '!=', '')
                ->where('address1', $this->address1)
        // name
                ->whereNotNull('name')
                ->where('name', '!=', '')
                ->where('name', $this->name);
        })->first();
    }

    public function scopeWhereAddress(Builder $builder, array $address): Builder
    {
        return $builder->where('name', $address['name'])
            ->where('zip', $address['zip'])
            ->where('address1', $address['address1']);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns(): array
    {
        return [
            'company',
            'name',
            'email',
            'phone',
            'fax',
            'address1',
            'address2',
            'address3',
            'city',
            'province',
            'province_code',
            'zip',
            'country',
            'country_code',
            'label',
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    public function toShipStationAddress(): \App\SDKs\ShipStation\Model\Address
    {
        $address = new \App\SDKs\ShipStation\Model\Address();
        $address->name = $this->name ?? $this->company;
        $address->company = $this->company;
        $address->street1 = $this->address1;
        $address->street2 = $this->address2;
        $address->street3 = $this->address3;
        $address->city = $this->city;
        $address->state = $this->province;
        $address->postalCode = $this->zip;
        $address->country = $this->country_code;
        $address->phone = $this->phone;

        return $address;
    }

    public function toStarshipitAddress(): DestinationDetails
    {
        $destinationDetails = new DestinationDetails();
        $destinationDetails->name = $this->name;
        $destinationDetails->email = $this->email;
        $destinationDetails->phone = $this->phone;
        $destinationDetails->building = "$this->address2 $this->address3";
        $destinationDetails->company = $this->company;
        $destinationDetails->street = $this->address1;
        $destinationDetails->suburb = $this->city;
        $destinationDetails->city = $this->city;
        $destinationDetails->state = $this->province;
        $destinationDetails->post_code = $this->zip;
        $destinationDetails->country = $this->country;

        return $destinationDetails;
    }

    public function resetPii()
    {
        $placeholder = '****Redacted****';

        DB::table('addresses')->whereId($this->id)->update([
            'name' => $placeholder,
            'email' => $placeholder,
            'phone' => $placeholder,
            'address1' => $placeholder,
            'address2' => $placeholder,
            'address3' => $placeholder,
        ]);

        $this->save();
    }
}
