<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Enums\AccountingTransactionTypeEnum;
use App\Exceptions\InvalidDataTableParametersException;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasLink;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\IntegrationInstanceRepository;
use App\Services\Accounting\AccountingIntegration;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Xero\Entities\XeroTransaction;

/**
 * @property int $id
 * @property Carbon $transaction_date
 * @property string $type
 * @property string $name
 * @property string $email
 * @property float $total
 * @property string $currency_code
 * @property float $currency_rate
 * @property string $reference
 * @property bool|null $is_tax_included
 * @property int $tax_rate_id
 * @property float $tax_amount
 * @property int $link_id
 * @property string $link_type
 * @property bool $is_batchable
 * @property bool $is_locked
 * @property int $accounting_integration_id
 * @property string $accounting_integration_type
 * @property Carbon $submission_started_at
 * @property int $parent_id
 * @property bool $is_sync_enabled
 * @property array $last_sync_error
 * @property-read AccountingIntegration $accountingIntegration
 * @property-read AccountingTransaction $parent
 * @property-read Currency $currency
 * @property-read Collection|AccountingTransaction[] $children
 * @property-read Collection|AccountingTransactionLine[] $accountingTransactionLines
 * @property-read string $status
 * @property FinancialDocumentInterface $link
 * @property TaxRate $taxRate*
 * @property Carbon $allocated_at
 * @property Carbon $updated_at
 * @property Carbon $created_at
 */
class AccountingTransaction extends Model implements Filterable, Sortable, UniqueFieldsInterface
{
    use HasFactory, HasFilters, HasLink, HasSort;

    const TYPE_SALES_ORDER_INVOICE = 'sales_order_invoice';

    const TYPE_BATCH_SALES_ORDER_INVOICE = 'batch_sales_order_invoice';

    const TYPE_SALES_ORDER_FULFILLMENT = 'sales_order_fulfillment';

    const TYPE_PURCHASE_ORDER = 'purchase_order';

    const TYPE_PURCHASE_ORDER_INVOICE = 'purchase_order_invoice';

    const TYPE_PURCHASE_ORDER_RECEIPT = 'purchase_order_receipt';

    const TYPE_INVENTORY_ADJUSTMENT = 'inventory_adjustment';

    const TYPE_CUSTOMER_RETURN = 'customer_return';

    const TYPE_SALES_CREDIT = 'sales_credit';

    const TYPE_STOCK_TAKE = 'stock_take';

    const TYPE_WAREHOUSE_TRANSFER_SHIPMENT = 'warehouse_transfer_shipment';

    const TYPE_WAREHOUSE_TRANSFER_RECEIPT = 'warehouse_transfer_receipt';

    const TYPE_RECEIVING_DISCREPANCY = 'receiving_discrepancy';

    const TYPES = [
        self::TYPE_SALES_ORDER_INVOICE,
        self::TYPE_SALES_ORDER_FULFILLMENT,
        self::TYPE_PURCHASE_ORDER,
        self::TYPE_PURCHASE_ORDER_INVOICE,
        self::TYPE_PURCHASE_ORDER_RECEIPT,
        self::TYPE_INVENTORY_ADJUSTMENT,
        self::TYPE_CUSTOMER_RETURN,
        self::TYPE_SALES_CREDIT,
        self::TYPE_STOCK_TAKE,
    ];

    const TYPES_WITH_PAYMENT = [
        self::TYPE_SALES_ORDER_INVOICE,
        self::TYPE_PURCHASE_ORDER_INVOICE,
        self::TYPE_SALES_CREDIT,
    ];

    const TYPES_INVOICE = [
        self::TYPE_SALES_ORDER_INVOICE,
        self::TYPE_PURCHASE_ORDER_INVOICE,
        self::TYPE_BATCH_SALES_ORDER_INVOICE,
    ];

    const TYPES_JOURNAL = [
        self::TYPE_SALES_ORDER_FULFILLMENT,
        self::TYPE_PURCHASE_ORDER_RECEIPT,
        self::TYPE_INVENTORY_ADJUSTMENT,
        self::TYPE_CUSTOMER_RETURN,
        self::TYPE_STOCK_TAKE,
        self::TYPE_WAREHOUSE_TRANSFER_SHIPMENT,
        self::TYPE_WAREHOUSE_TRANSFER_RECEIPT,
        self::TYPE_RECEIVING_DISCREPANCY,
    ];

    const TYPES_NON_POSTING = [
        self::TYPE_PURCHASE_ORDER,
    ];

    const STATUS_TO_SYNC = 'toSync';

    const STATUS_SYNCED = 'synced';

    const STATUS_HAS_ERRORS = 'hasErrors';

    const STATUS_SUBMISSION_IN_PROGRESS = 'submissionInProgress';

    const STATUS_INELIGIBLE_TO_SYNC = 'ineligibleToSync';

    const STATUSES = [
        self::STATUS_TO_SYNC,
        self::STATUS_SYNCED,
        self::STATUS_HAS_ERRORS,
        self::STATUS_INELIGIBLE_TO_SYNC,
    ];

    protected $casts = [
        'transaction_date' => 'datetime',
        'tax_amount' => 'float',
        'is_batchable' => 'boolean',
        //'is_tax_included' => 'boolean', // should not be casted to boolean as it can be null
        'is_locked' => 'boolean',
        'last_sync_error' => 'array',
        'is_sync_enabled' => 'boolean'
    ];

    protected $fillable = [
        'id',
        'transaction_date',
        'type',
        'name',
        'email',
        'total',
        'currency_code',
        'currency_rate',
        'link_id',
        'link_type',
        'is_batchable',
        'accounting_integration_id',
        'accounting_integration_type',
        'reference',
        'is_tax_included',
        'tax_amount',
        'tax_rate_id',
        'updated_at',
        'parent_id',
        'allocated_at',
        'submission_started_at',
        'last_sync_error',
        'is_locked',
        'is_sync_enabled'
    ];

    public static function getUniqueFields(): array
    {
        return [
            'link_id',
            'link_type',
            'type',
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function accountingTransactionLines(): HasMany
    {
        return $this->hasMany(AccountingTransactionLine::class);
    }

    public function link(): MorphTo
    {
        return $this->morphTo('link');
    }

    public function taxRate(): BelongsTo
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function accountingIntegration(): MorphTo|AccountingIntegration
    {
        return $this->morphTo('accountingIntegration');
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'code');
    }

    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function saveLastSyncError(?array $errors): void
    {
        $this->submission_started_at = null;
        $this->last_sync_error = $errors;
        $this->timestamps = false;
        $this->save();
    }

    public function availableColumns()
    {
        return config('data_table.accounting_transaction.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['reference'];
    }

    public function getStatusAttribute(): string
    {
        if ($this->is_batchable) {
            return self::STATUS_INELIGIBLE_TO_SYNC;
        }
        if ($this->submission_started_at) {
            return self::STATUS_SUBMISSION_IN_PROGRESS;
        }
        if ($this->last_sync_error) {
            return self::STATUS_HAS_ERRORS;
        }

        if ($this->accountingIntegration && !$this->last_sync_error) {
            return self::STATUS_SYNCED;
        }

        if (app(IntegrationInstanceRepository::class)->getAccountingInstance() && $this->transaction_date < app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']) {
            return self::STATUS_INELIGIBLE_TO_SYNC;
        }

        if (
            ($this->accountingIntegration && $this->accountingIntegration->status != 'PAID' && $this->updated_at > $this->accountingIntegration->updated_at) ||
            !$this->accountingIntegration
        ) {
            if (app(IntegrationInstanceRepository::class)->getAccountingInstance()) {
                if ($this->transaction_date >= app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']) {
                    return self::STATUS_TO_SYNC;
                }
            }
        }

        return 'Unknown';
    }

    /**
     * @throws Exception
     */
    public function scopeFilterStatus(Builder $query, array $relation, string $operator, ?string $value, $conjunction): Builder
    {
        if (!$value || $operator == '!=') {
            throw new InvalidDataTableParametersException('You can only filter by "is" for status');
        }
        switch ($value) {
            case 'toSync':
                $query->where(function (Builder $builder) {
                    $builder->whereHas('accountingIntegration', function (Builder $query) {
                        // Transactions that are paid don't need syncing
                        $query->where('status', '!=', 'PAID');
                        $query->whereColumn('accounting_transactions.updated_at', '>', 'updated_at');
                    })
                        ->orWhereNull('accounting_integration_id');
                });
                if (app(IntegrationInstanceRepository::class)->getAccountingInstance()) {
                    $query->where('transaction_date', '>=', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']);
                }
                $query->where('is_batchable', false);

                return $query;
            case 'synced':
                return $query->whereHas('accountingIntegration', function (Builder $query) {
                    $query->whereNull('last_error');
                });
            case 'hasErrors':
                return $query->whereHas('accountingIntegration', function (Builder $query) {
                    $query->whereNotNull('last_error');
                });
            case 'ineligibleToSync':
                if (app(IntegrationInstanceRepository::class)->getAccountingInstance()) {
                    return $query->where('transaction_date', '<', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']);
                }

                return $query;
            default:
                return $query;
        }
    }

    public function scopeFilterAccountingIntegration(Builder $query, array $relation, string $operator, $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'whereHasMorph' : 'orWhereHasMorph';

        $query->{$function}($relation['name'], [XeroTransaction::class], function ($q) use ($relation, $operator, $value) {
            $q->filterKey($relation['key'], $operator, $value);
        });

        if (in_array($operator, ['!=', 'doesNotContain', 'isEmpty'])) {
            $query->orWhereNull($this->{$relation['name']}()->getForeignKeyName());
        }

        return $query;
    }

    public function scopeFilterPurchaseReceiptDoesntHavePurchaseInvoicesButBookedToAccruedPurchases(Builder $query, string $operator, $value, $conjunction): Builder
    {
        if (!$value) return $query;

        return $query->where('type', AccountingTransactionTypeEnum::PURCHASE_ORDER_RECEIPT)
            ->whereHasMorph('link', [PurchaseOrderShipmentReceipt::class], function (Builder $query) {
                $query->whereHas('purchaseOrderShipment.purchaseOrder', function (Builder $query) {
                    $query->whereDoesntHave('purchaseInvoices', function (Builder $query) {
                        $query->whereHas('accountingTransaction');
                    });
                });
            })
            ->whereHas('accountingTransactionLines', function (Builder $query) {
                $query->whereHas('nominalCode', function (Builder $query) {
                    $query->where('name', 'Accrued Purchases');
                });
            });
    }

    public function scopeWithSalesOrders(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesOrder::class, $ids);
    }

    public function scopeWithSalesCredits(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesCredit::class, $ids);
    }

    public function scopeWithSalesOrderFulfillments(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesOrderFulfillment::class, $ids);
    }

    public function scopeWithPurchaseOrders(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseOrder::class, $ids);
    }

    public function scopeWithPurchaseInvoices(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseInvoice::class, $ids);
    }

    public function scopeWithPurchaseOrderReceipts(Builder $query, array|int $ids)
    {
        return $query->withLink(PurchaseOrderShipmentReceipt::class, $ids);
    }

    public function scopeWithInventoryAdjustments(Builder $query, array|int $ids)
    {
        return $query->withLink(InventoryAdjustment::class, $ids);
    }

    public function scopeWithCustomerReturns(Builder $query, array|int $ids)
    {
        return $query->withLink(SalesCreditReturn::class, $ids);
    }

    public function scopeWithStockTakes(Builder $query, array|int $ids)
    {
        return $query->withLink(StockTake::class, $ids);
    }

    public function scopeWithWarehouseTransferShipments(Builder $query, array|int $ids)
    {
        return $query->withLink(WarehouseTransferShipment::class, $ids);
    }

    public function scopeWithWarehouseTransferReceipts(Builder $query, array|int $ids)
    {
        return $query->withLink(WarehouseTransferShipmentReceipt::class, $ids);
    }

    public function scopeTrailingDays(Builder $query, int $trailing_days): Builder
    {
        return $query->where('transaction_date', '>=', Carbon::now()->subDays($trailing_days));
    }

    public function scopeStartDate(Builder $query, $date): Builder
    {
        return $query->where('transaction_date', '>=', Carbon::parse($date));
    }

    public function scopeEndDate(Builder $query, $date): Builder
    {
        return $query->where('transaction_date', '<', Carbon::parse($date));
    }

    public function delete(): ?bool
    {
        $this->accountingTransactionLines()->delete();
        $this->children->each(function (self $accountingTransaction) {
            $accountingTransaction->parent_id = null;
            $accountingTransaction->update();
        });

        return parent::delete();
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function scopeSortAccountingIntegration(Builder $builder, array $relation, bool $ascending)
    {
        switch (app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration->name) {
            case Integration::NAME_XERO:
                $className = XeroTransaction::class;
                $tableName = app(XeroTransaction::class)->getTable();
                break;
            default:
                return $builder->whereHas('accountingIntegration', function (Builder $builder) use ($relation, $ascending) {
                    $builder->orderBy($relation['key'], $ascending ? 'asc' : 'desc');
                });
        }

        return $builder->leftJoin($tableName, function ($join) use ($tableName, $className) {
            $join->on('accounting_transactions.accounting_integration_id', '=', $tableName.'.id')
                ->where('accounting_transactions.accounting_integration_type', '=', $className);
        })
            ->orderBy(DB::raw('ISNULL('.$tableName.'.'.$relation['key'].'), '.$tableName.'.'.$relation['key']), $ascending ? 'asc' : 'desc')
            ->select('accounting_transactions.*', $tableName.'.'.$relation['key'].' as integration_'.$relation['key']);
    }

    public function scopeSortParent(Builder $builder, array $relation, bool $ascending)
    {
        $selfTableName = (new self())->getTable();

        $builder->join($selfTableName.' as parent', function ($join) use ($selfTableName) {
            $join->on($selfTableName.'.parent_id', 'parent.id');
        })
            ->orderBy('parent.'.$relation['key'], $ascending ? 'asc' : 'desc');

        return $builder;
    }
}
