<?php

namespace App\Managers;

use App\Data\AdjustmentForReceiptData;
use App\Data\BlemishedProductData;
use App\Data\CreateInventoryAdjustmentFromBlemishedProductData;
use App\Data\WarehouseTransferReceiptBlemishedData;
use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Events\WarehouseTransferInitiated;
use App\Events\WarehouseTransferReceived;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferHasNoProductsException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferOpenException;
use App\Helpers;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductBlemished;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Repositories\WarehouseTransferRepository;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\Receipts\CreatesAdjustmentForReceipt;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonRemovalOrderManager;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

/**
 * Class TransferProcessor.
 */
class WarehouseTransferManager
{

    use CreatesAdjustmentForReceipt;

    public function __construct(
        private readonly WarehouseTransferRepository $warehouseTransfers,
        private readonly BlemishedProductManager $blemishedProductManager
    )
    {
    }

    /**
     * Initiates warehouse transfer.
     *
     *
     * @throws Throwable
     */
    public function initiateTransfer(array $payload): WarehouseTransfer
    {
        $fromWarehouse = Warehouse::with([])->find($payload['from_warehouse_id']);
        if ($fromWarehouse->isAmazonFBA()) {
            // TODO: Move this logic to a manager within the Amazon module
            //return ( new RemovalOrderWarehouseTransfer($payload) )->handle();
            return (new AmazonRemovalOrderManager(AmazonIntegrationInstance::find($fromWarehouse->integration_instance_id)))->process($payload['order_id'], $payload['to_warehouse_id'])->first();
        }
        /** @var Warehouse $toWarehouse */
        $toWarehouse = Warehouse::with([])->find($payload['to_warehouse_id']);
        if ($toWarehouse->isAmazonFBA()) {
            // TODO: Rethink this flow, the shipment exists here, but the warehouse transfer doesn't.  Maybe this shouldn't even be an option in the UI?
            //  Better to create from inbound section?  Or maybe before we open the new warehouse transfer drawer.
            return (new AmazonInboundManager(AmazonIntegrationInstance::find($toWarehouse->integration_instance_id)))->processWarehouseTransfer($payload['shipment_id'], $payload['from_warehouse_id']);
            // Old way:
            // return ( new InboundShipmentWarehouseTransfer($payload) )->handle();
        }

        // We run the transfers in a migration so we can rollback if needed
        return DB::transaction(function () use ($payload) {
            // We create the warehouse transfer
            $transfer = $this->warehouseTransfers->createTransfer($payload);
            // We set the warehouse number.
            $transfer->setTransferNumber($this->makeTransferNumber($transfer->id));

            // Create lines if available
            if (isset($payload['products'])) {
                // We aggregate the products to ensure that only unique products end
                // up in the warehouse transfer
                $payload['products'] = $this->aggregateProducts($payload['products']);
                // Create warehouse transfer lines
                foreach ($payload['products'] as $product) {
                    // Check if the source warehouse has enough quantities on stock for the
                    // transfer.
                    //                    $this->checkStockLevelForProduct($product['id'], $product['quantity'], $transfer->fromWarehouse);
                    $this->warehouseTransfers->createWarehouseTransferLine($transfer, $product['id'], $product['quantity']);
                }
            }

            // Return the warehouse transfer
            return $transfer;
        });
    }

    private function aggregateProducts(array $products): array
    {
        $uniqueProductIds = array_unique(array_map(function ($product) {
            return $product['id'];
        }, $products));
        $uniqueProducts = [];
        foreach ($uniqueProductIds as $productId) {
            $matched = array_filter($products, function ($product) use ($productId) {
                return $product['id'] === $productId;
            });

            $uniqueProducts[] = [
                'id' => $productId,
                'quantity' => array_sum(array_map(function ($matchedProduct) {
                    return $matchedProduct['quantity'];
                }, $matched)),
            ];
        }

        return $uniqueProducts;
    }

    /**
     * @throws InsufficientStockException
     */
    private function checkStockLevelForProduct($productId, $quantity, Warehouse $fromWarehouse)
    {
        if ($fromWarehouse->currentStockLevelForProduct($productId) < $quantity) {
            throw new InsufficientStockException($productId);
        }
    }

    /**
     * Note: Only use this method if you want to create a single warehouse shipment to cover all the lines in a
     * warehouse transfer
     *
     * TODO: This is so confusing... why is it accepting data?  Why not just use what is already saved in the warehouse transfer?
     *
     * @throws WarehouseTransferHasNoProductsException
     * @throws WarehouseTransferOpenException|InsufficientStockException
     */
    public function openWarehouseTransfer(WarehouseTransfer $transfer, array $data): WarehouseTransfer
    {
        // Warehouse transfer must be draft
        if (! $transfer->isDraft()) {
            throw new WarehouseTransferOpenException($transfer);
        }

        // Update transfer data
        $transfer->fill($data);
        $transfer->save();

        // Create warehouse transfer lines for products, if any
        foreach (($data['products'] ?? []) as $product) {
            // Check if the source warehouse has enough quantities on stock for the
            // transfer.
            //            $this->checkStockLevelForProduct($product['id'], $product['quantity'], $transfer->fromWarehouse);
            $this->warehouseTransfers->createWarehouseTransferLine($transfer, $product['id'], $product['quantity']);
        }

        // Warehouse transfer must have products to be open
        if ($transfer->warehouseTransferLines()->count() === 0) {
            throw new WarehouseTransferHasNoProductsException;
        }

        // We go through the Warehouse Transfer lines and create inventory movements
        $fromWarehouseLocation = $transfer->fromWarehouse->defaultLocation;
        $warehouseTransferLines = $transfer->warehouseTransferLines;

        // Get the shipment payload or create the minimum required for creating a shipment (for Phase A1).
        $shipmentPayload = $data['shipment'] ?? ['shipment_date' => $transfer->transfer_date];

        // Create shipment
        $shipment = $this->warehouseTransfers->createTransferShipment($transfer, $shipmentPayload);

        // This creates a shipment line for every warehouse transfer line.
        foreach ($warehouseTransferLines as $warehouseTransferLine) {
            // For Phase A1, We handle shipment with its inventory movements
            // when opening the warehouse transfer.
            $quantityShipped = $warehouseTransferLine->quantity;
            $this->createShipmentLineForTransferLine($warehouseTransferLine, $transfer, $quantityShipped, $shipment);
        }

        // We handle the shipment status of the warehouse transfer
        $this->setTransferShipmentStatus($transfer, $shipment->shipped_at);

        // We open the warehouse transfer
        $transfer = $transfer->open();

        // Broad initiation of warehouse transfer
        event(new WarehouseTransferInitiated($transfer));

        return $transfer;
    }

    /**
     * @throws Exception
     */
    public function makeTransferDraft(WarehouseTransfer $transfer): WarehouseTransfer
    {
        return DB::transaction(function () use ($transfer) {
            // Remove shipment and all receipts
            if ($transfer->shipment) {
                $transfer->shipments->each(function (WarehouseTransferShipment $shipment) {
                    $shipment->delete();
                });
            }

            // Set transfer statuses
            $transfer->transfer_status = WarehouseTransfer::TRANSFER_STATUS_DRAFT;
            $transfer->shipment_status = WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_UNSHIPPED;
            $transfer->receipt_status = WarehouseTransfer::TRANSFER_RECEIPT_STATUS_UNRECEIVED;
            $transfer->fully_shipped_at = null;
            $transfer->fully_received_at = null;
            $transfer->save();

            return $transfer;
        });
    }

    public function updateTransfer(WarehouseTransfer $transfer, array $data): bool
    {
        return DB::transaction(function () use ($transfer, $data) {
            // We update the inventory movement dates and warehouses
            //      $this->updateMovementDatesAndWarehouses( $transfer, $data );

            // If changing products
            if (isset($data['products'])) {
                $this->updateTransferProducts($transfer, $data);
            }

            // Update transfer
            $transfer->fill($data);

            return $transfer->save();
        });
    }

    /**
     * Updates the products of the warehouse transfer.
     *
     *
     * @throws InsufficientStockException
     */
    public function updateTransferProducts(WarehouseTransfer $transfer, array $data, bool $sync = true)
    {
        // Aggregate product quantities
        $products = $this->aggregateProducts($data['products']);

        $transferProductIds = $transfer->warehouseTransferLines()->pluck('product_id')->toArray();
        $newProductIds = array_map(function ($product) {
            return $product['id'];
        }, $products);

        $keptProductIds = array_intersect($transferProductIds, $newProductIds);
        $additionalProductIds = array_diff($newProductIds, $keptProductIds);
        $removedProductIds = array_diff($transferProductIds, $keptProductIds);

        // Delete the transfer lines for the removed products
        if ($sync) {
            $transfer->warehouseTransferLines()->whereIn('product_id', $removedProductIds)->each(function (WarehouseTransferLine $line) {
                $line->delete();
            });
        }

        // Get the "from" warehouse
        if ($this->isChangingWarehouse($transfer, $data)) {
            /** @var Warehouse $fromWarehouse */
            $fromWarehouse = Warehouse::with([])->findOrFail($data['from_warehouse_id']);
            if (! $transfer->isDraft()) {
                if ($transfer->shipmentReceipts()->count() > 0) {
                    /**
                     * Cannot change warehouse when some products
                     * are already received.
                     */
                    throw new InvalidArgumentException('Cannot change warehouse of warehouse transfers with receipts.');
                }

                if ($transfer->shipment) {
                    $transfer->shipment->delete();
                }
            }

            $transfer->from_warehouse_id = $fromWarehouse->id;
            $transfer->save();

            $shipmentPayload = $data['shipment'] ?? ['shipment_date' => $transfer->transfer_date];

            // Create shipment
            $shipment = $this->warehouseTransfers->createTransferShipment($transfer, $shipmentPayload);
        } else {
            $fromWarehouse = $transfer->fromWarehouse;
            $shipment = null;
        }

        // Update quantities for kept products if changed
        $transfer->warehouseTransferLines()
            ->whereIn('product_id', $keptProductIds)
            ->each(function (WarehouseTransferLine $transferLine) use ($products, $transfer, $data, $shipment) {
                $product = $this->matchProductInPayload($transferLine->product_id, $products);
                $manager = InventoryManager::with(
                    $transferLine->warehouseTransfer->from_warehouse_id,
                    $transferLine->product
                );

                // Make sure product has enough stock if the transfer isn't draft
                if (! $transfer->isDraft()) {
                    if ($this->isChangingWarehouse($transfer, $data, 'from')) {
                        /**
                         * The user just changed the origin warehouse, so
                         * we create this as a new pull from the new warehouse
                         */
                        $quantityShipped = $transferLine->quantity;
                        $this->createShipmentLineForTransferLine($transferLine, $transfer, $quantityShipped, $shipment);
                    } else {
                        /**
                         * The user isn't changing the origin warehouse.
                         * We adjust inventory for quantity changes.
                         */
                        $diff = $product['quantity'] - $transferLine->quantity;
                        if ($diff > 0) {
                            // Quantity is being increased.
                            $manager->increaseNegativeEventQty($diff, $transferLine->shipmentLine);
                        } elseif ($diff < 0) {
                            // Quantity is being reduced
                            $manager->decreaseNegativeEventQty(abs($diff), $transferLine->shipmentLine);
                        }
                    }

                    /**
                     * When the destination warehouse changes, we simply adjust
                     * the in-transit movement warehouse to reflect the new warehouse.
                     * Note that at this point, no inventory movements are made at the
                     * destination warehouse.
                     */
                    if ($this->isChangingWarehouse($transfer, $data, 'to')) {
                        $inTransitMovement = $transferLine->shipmentLine
                            ->inventoryMovements()
                            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
                            ->where('quantity', '>', 0)
                            ->first();
                        if ($inTransitMovement) {
                            $inTransitMovement->update([
                                'warehouse_id' => $data['to_warehouse_id'],
                            ]);
                        }
                    }
                }

                // Update the transfer line quantity.
                $transferLine->quantity = $product['quantity'];
                $transferLine->save();
            });

        // Create new warehouse transfer lines for added products
        $additionalProducts = collect($data['products'])->whereIn('id', $additionalProductIds);
        $fromWarehouseLocation = isset($data['from_warehouse_location_id']) ?
      $fromWarehouse->warehouseLocations()->findOrFail($data['from_warehouse_location_id']) : $fromWarehouse->defaultLocation;

        foreach ($additionalProducts as $product) {
            // Check if the source warehouse has enough quantities on stock for the
            // transfer.
            $transferLine = $this->warehouseTransfers->createWarehouseTransferLine($transfer, $product['id'], $product['quantity']);

            if (! $transfer->isDraft()) {
                // For Phase A1, We handle shipment with its inventory movements
                // when opening the warehouse transfer.
                $quantityShipped = $transferLine->quantity;
                $this->createShipmentLineForTransferLine($transferLine, $transfer, $quantityShipped, $transfer->shipment);
            }
        }
    }

    private function isChangingWarehouse(WarehouseTransfer $transfer, array $data, string $type = 'from'): bool
    {
        if ($type === 'from') {
            return isset($data['from_warehouse_id']) && $data['from_warehouse_id'] !== $transfer->from_warehouse_id;
        } else {
            return isset($data['to_warehouse_id']) && $data['to_warehouse_id'] !== $transfer->to_warehouse_id;
        }
    }

    private function matchProductInPayload($productId, array $products): ?array
    {
        $matched = array_filter($products, function ($product) use ($productId) {
            return $product['id'] === $productId;
        });
        if (! empty($matched)) {
            return array_values($matched)[0];
        }

        return null;
    }

    /**
     * @throws InsufficientStockException
     */
    private function createShipmentLineForTransferLine(
        WarehouseTransferLine $warehouseTransferLine,
        WarehouseTransfer $warehouseTransfer,
        $quantityShipped,
        WarehouseTransferShipment $shipment
    ): WarehouseTransferShipment {
        // Create the shipment line
        $shipmentLine = $this->warehouseTransfers->createTransferShipmentLine($shipment, $warehouseTransferLine->id, $quantityShipped);

        InventoryManager::with(
            $shipment->warehouseTransfer->from_warehouse_id,
            $warehouseTransferLine->product
        )->takeFromStock($warehouseTransferLine->quantity, $shipmentLine);

        return $shipment;
    }

    public function setTransferShipmentStatus(WarehouseTransfer $warehouseTransfer, $shipmentDate = null): void
    {
        // Set the shipment status based on total shipped quantity for all lines
        if ($warehouseTransfer->total_shipped === ($warehouseTransfer->total_quantity - $warehouseTransfer->total_shipped_before_start_date)) {
            // We indicate that the transfer is fully shipped and we use the
            // supplied shipping date since this will be the last shipment.
            $warehouseTransfer->totallyShipped($shipmentDate);
        } elseif ($warehouseTransfer->total_shipped === 0) {
            $warehouseTransfer->shipment_status = WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_UNSHIPPED;
            $warehouseTransfer->transfer_status = WarehouseTransfer::TRANSFER_STATUS_OPEN;
            $warehouseTransfer->save();
        } else {
            // The transfer is only partially shipped.
            $warehouseTransfer->partiallyShipped();
        }
    }

    /**
     * Receives warehouse transfer shipment.
     *
     *
     * @throws NotOpenWarehouseTransferException
     * @throws Throwable
     */
    public function receiveShipment(WarehouseTransfer $warehouseTransfer, WarehouseTransferReceiptData $data): WarehouseTransferShipmentReceipt|WarehouseTransfer
    {
        // Warehouse transfer must be open for shipment to be received
        if ($warehouseTransfer->isDraft() || $warehouseTransfer->isClosed()) {
            if (!$warehouseTransfer->toWarehouse->isAmazonFBA()) {
                throw new NotOpenWarehouseTransferException($warehouseTransfer);
            }
        }

        // Run the receipt in a transaction, so they're rolled back if something goes wrong
        return DB::transaction(function () use ($warehouseTransfer, $data) {
            // We get the shipment id and use it to create the receipt
            if ($data->shipment_id instanceof Optional) {
                $data->shipment_id = $warehouseTransfer->shipment->id;
            }

            /** @var WarehouseTransferShipmentReceipt $receipt */
            $receipt = $this->warehouseTransfers->createTransferShipmentReceipt($data->shipment_id, $data->receipt_date);

            // We receive the transfer shipment for each product
            $transferLines = $warehouseTransfer
                ->warehouseTransferLines()
                ->whereIn('product_id', array_map(function ($product) {
                    return $product['id'];
                }, $data->products->toArray()))
                ->get()
                ->keyBy('product_id');

            $blemished = $data->blemished instanceof Optional ? collect() : $data->blemished->toCollection();
            $receiptLinesWithBlemishedProducts = [];

            /** @var WarehouseTransferReceiptProductData $product */
            foreach ($data->products as $product) {
                //Log::debug('Warehouse Transfer Receive Shipment Product', $product);
                $warehouseTransferLine = $transferLines->get($product->id);

                if($blemished->isEmpty() || $blemished->where('product_id', $product->id)->isEmpty()){
                    // Receive shipment for line without blemished skus
                    /** @var WarehouseTransferLine $warehouseTransferLine */
                    $this->receiveShipmentForTransferLine(
                        warehouseTransferLine: $warehouseTransferLine,
                        receipt: $receipt,
                        product: $product->toArray()
                    );
                } else {
                    // Receive shipment for line with blemished skus.
                    // We don't apply stock for blemished products yet.
                    /** @var WarehouseTransferLine $warehouseTransferLine */
                    $receiptLinesWithBlemishedProducts[] = $this->receiveShipmentForTransferLine(
                        warehouseTransferLine: $warehouseTransferLine,
                        receipt: $receipt,
                        product: $product->toArray(),
                        autoApplyStock: false
                    );
                }
            }

            // Handle blemished products in the receipt.
            if($blemished->isNotEmpty()){
                $this->handleBlemishedProductsInReceipt(
                    $blemished,
                    $receipt,
                    $receiptLinesWithBlemishedProducts
                );
            }

            // If the receipt has no lines, delete it.
            if ($receipt->receiptLines()->count() === 0) {
                $receipt->delete();
                return $warehouseTransfer;
            } else {
                // We set the status of the shipment
                $this->setTransferReceiptStatus($warehouseTransfer, $data->receipt_date);

                event(new WarehouseTransferReceived($receipt));

                return $receipt;
            }
        });
    }


    private function handleBlemishedProductsInReceipt(
        Collection $blemished,
        WarehouseTransferShipmentReceipt $receipt,
        array $receiptLinesWithBlemishedProducts
    ): void {

        $receiptLines = $receipt->receiptLines()->get()->keyBy(function (WarehouseTransferShipmentReceiptLine $receiptLine) {
            return $receiptLine->shipmentLine->warehouseTransferLine->product_id;
        });

        $this->generateBlemishedProductDataFromWarehouseTransferReceipt($blemished, $receipt, $receiptLines)->each(function (BlemishedProductData $data) use ($receipt) {
            $this->blemishedProductManager->createBlemishedProduct($data, true);
        });
    }

    private function generateBlemishedProductDataFromWarehouseTransferReceipt(Collection $blemished, WarehouseTransferShipmentReceipt $receipt, Collection $receiptLines): DataCollection {
        return BlemishedProductData::collection($blemished->map(function (WarehouseTransferReceiptBlemishedData $data) use ($receipt, $receiptLines) {
            $product = Product::find($data->product_id);
            /** @var WarehouseTransferShipmentReceiptLine $receiptLine */
            $receiptLine = $receiptLines->get($product->id);
            return BlemishedProductData::from([
               'blemished_sku' => $data->sku,
                'derived_from_product_id' => $data->product_id,
                'condition' => $data->condition,
                'reference' => $data->reference,
                'adjustment_data' => CreateInventoryAdjustmentFromBlemishedProductData::from([
                    'adjustment_date' => $receipt->received_at,
                    'warehouse_id' => $receipt->shipment->warehouseTransfer->to_warehouse_id,
                    'quantity' => 1,
                    'unit_cost' => $product->getUnitCostAtWarehouse($receipt->shipment->warehouseTransfer->to_warehouse_id),
                    'notes' => 'Create blemished product from warehouse transfer receipt for ' . $receipt->shipment->warehouseTransfer->warehouse_transfer_number,
                    'link_type' => WarehouseTransferLine::class,
                    'link_id' => $receiptLine->shipmentLine->warehouseTransferLine->id,
                ]),
                'adjust_inventory' => true,
                'adjust_inventory_for_original_product' => true,
            ]);
        }));
    }

    /**
     * @param  WarehouseTransfer  $warehouseTransfer
     * @param  WarehouseTransferShipmentReceipt  $receipt
     * @param  array  $data
     * @return mixed
     * @throws NotOpenWarehouseTransferException
     * @throws Throwable
     */
    public function updateShipmentReceipt(
        WarehouseTransfer $warehouseTransfer,
        WarehouseTransferShipmentReceipt $receipt,
        array $data
    )
    {
        // Warehouse transfer must be open for shipment to be received
        if ($warehouseTransfer->isDraft() || $warehouseTransfer->isClosed()) {
            throw new NotOpenWarehouseTransferException($warehouseTransfer);
        }

        // Run the receipt in a transaction, so they're rolled back if something goes wrong
        return DB::transaction(function () use ($warehouseTransfer, $receipt, $data) {
            // Update receipt date is available
            if (! empty($data['receipt_date'] ?? [])) {
                $receipt->update(['received_at' => $data['receipt_date']]);
            }

            // We get the products in the original receipt and figure out
            // removed products from the receipt
            $originalProducts = $receipt->receiptLines->map(function (WarehouseTransferShipmentReceiptLine $receiptLine) {
                return $receiptLine->shipmentLine->warehouseTransferLine->product->id;
            })->toArray();

            $productIds = array_map(function ($product) {
                return $product['id'];
            }, $data['products']);
            $keptProductIds = array_intersect($originalProducts, $productIds);
            $removedProductIds = array_diff($originalProducts, $keptProductIds);

            // We remove the receipt lines for the removed products
            $receipt->receiptLines()->whereHas('shipmentLine.warehouseTransferLine', function (Builder $builder) use ($removedProductIds) {
                return $builder->whereIn('product_id', $removedProductIds);
            })->each(function (WarehouseTransferShipmentReceiptLine $line) {
                $line->delete(); // This ensures that inventory movements are deleted too
            });

            // For the kept products, we update the receipt quantities and the inventory movements
            $keptProducts = array_filter($data['products'], function ($product) use ($keptProductIds) {
                return in_array($product['id'], $keptProductIds);
            });

            foreach ($keptProducts as $product) {
                /** @var WarehouseTransferShipmentReceiptLine $receiptLine */
                $receiptLine = $receipt->receiptLines()->whereHas('shipmentLine.warehouseTransferLine', function (Builder $builder) use ($product) {
                    return $builder->where('product_id', $product['id']);
                })->firstOrFail();

                $receiptLine->quantity = $product['quantity'];
                $receiptLine->inventoryMovements()->update([
                    'quantity' => $product['quantity'],
                    'inventory_movement_date' => $receipt->received_at,
                ]);
                $receiptLine->save();
            }

            // We set the status of the shipment
            if ($warehouseTransfer->shipment->isFullyReceived()) {
                $warehouseTransfer->shipment->fullyReceived($data['receipt_date']);
            }

            //      event( new WarehouseTransferReceived( $receipt ) );

            return $this->setTransferReceiptStatus($warehouseTransfer, $data['receipt_date'] ?? null);
        });
    }

    /**
     * @param WarehouseTransfer $warehouseTransfer
     * @param WarehouseTransferShipmentReceipt $receipt
     * @return WarehouseTransfer
     * @throws Throwable
     */
    public function deleteShipmentReceipt(
        WarehouseTransfer $warehouseTransfer,
        WarehouseTransferShipmentReceipt $receipt
    ): WarehouseTransfer
    {
        // Run the receipt in a transaction so they're rolled back if something goes wrong
        return DB::transaction(function () use ($warehouseTransfer, $receipt) {
            // Receipt should be on the transfer
            $receipt->delete();

            // We set the status of the shipment
            if (! $warehouseTransfer->shipment->isFullyReceived()) {
                $warehouseTransfer->shipment->fully_received_at = null;
                $warehouseTransfer->shipment->save();
            }

            return $this->setTransferReceiptStatus($warehouseTransfer, now());
        });
    }

    /**
     * @param WarehouseTransfer $warehouseTransfer
     * @param $receiptDate
     * @return WarehouseTransfer
     */
    public function setTransferReceiptStatus(
        WarehouseTransfer $warehouseTransfer,
        $receiptDate = null
    ) : WarehouseTransfer
    {
        // We set the receipt status of the warehouse transfer based on the total quantities received.
        if ($warehouseTransfer->total_received === 0) {
            $warehouseTransfer->receipt_status = WarehouseTransfer::TRANSFER_RECEIPT_STATUS_UNRECEIVED;
            $warehouseTransfer->transfer_status = WarehouseTransfer::TRANSFER_STATUS_OPEN;
            $warehouseTransfer->save();
        } elseif ($warehouseTransfer->total_received === ($warehouseTransfer->total_quantity - $warehouseTransfer->total_shipped_before_start_date)) {
            // We indicate that the warehouse transfer is totally received and close the transfer
            $warehouseTransfer->totallyReceived($receiptDate ?? now());
            $warehouseTransfer = $warehouseTransfer->close();
        } else {
            // The transfer is only partially received, we set the receipt status
            $warehouseTransfer = $warehouseTransfer->partiallyReceived();
        }

        return $warehouseTransfer;
    }

    /**
     * @throws Exception|Throwable
     */
    public function receiveShipmentForTransferLine(
        WarehouseTransferLine $warehouseTransferLine,
        WarehouseTransferShipmentReceipt $receipt,
        array $product,
        bool $autoApplyStock = true
    ): WarehouseTransferShipmentReceiptLine|InventoryAdjustment {

        // For negative quantities, we create
        // negative adjustments for the warehouse transfer line
        if ($product['quantity'] < 0) {

            $existingReceiptFifoLayerIds = WarehouseTransferShipmentReceiptLine::with(['fifoLayers'])
                ->whereHas('shipmentLine', function (Builder $query) use ($warehouseTransferLine) {
                    $query->where('warehouse_transfer_line_id', $warehouseTransferLine->id);
                })
                // For now, we want to limit to 1 and the one with the biggest quantity so that it will be likely to cover the negative adjustment needs
                ->orderBy('quantity', 'desc')
                ->limit(1)
                ->get()
                ->pluck('fifoLayers')->flatten()->pluck('id')->toArray();

            if(empty($existingReceiptFifoLayerIds)){
                throw new Exception(
                    'No existing warehouse transfer receipt fifo layers found for negative receipt lines.'
                );
            }

            return $this->createAdjustment(
                AdjustmentForReceiptData::from([
                    'quantity' => $product['quantity'],
                    'warehouse_id' => $warehouseTransferLine->warehouseTransfer->to_warehouse_id,
                    'product_id' => $warehouseTransferLine->product_id,
                    'note' => 'Negative warehouse transfer receipt.',
                    'received_at' => $receipt->received_at,
                    'link_type' => WarehouseTransferLine::class,
                    'link_id' => $warehouseTransferLine->id,
                    'applicable_fifo_layer_ids' => $existingReceiptFifoLayerIds,
                ])
            );
        }

        // We create the shipment receipts
        $shipmentLine = $warehouseTransferLine->shipmentLine;

        // Create receipt line
        $receiptLine = $this->warehouseTransfers
            ->createWarehouseTransferReceiptLine($receipt->id, $shipmentLine->id, $product['quantity']);

        $inventoryManager = InventoryManager::with(
            $shipmentLine->warehouseTransferLine->warehouseTransfer->to_warehouse_id,
            $shipmentLine->warehouseTransferLine->product
        );

        if ($product['quantity'] > 0) {
            $inventoryManager->addToStock(
                quantity: $receiptLine->quantity,
                event: $receiptLine,
                syncStockNow: $autoApplyStock,
                autoApplyStock: $autoApplyStock
            );
        }

        $this->setTransferReceiptStatus($warehouseTransferLine->warehouseTransfer, $receipt->received_at);

        event(new WarehouseTransferReceived($receipt));

        return $receiptLine;
    }

    private function makeTransferNumber($transferId): string
    {
        $prefix = Setting::getValueByKey(Setting::KEY_WH_TRANSFER_PREFIX) ?? 'WT-';
        $padLength = Setting::getValueByKey(Setting::KEY_WH_TRANSFER_NUM_OF_DIGITS) ?? 4;

        return $prefix.str_pad($transferId, $padLength, '0', STR_PAD_LEFT);
    }

    public function duplicate(WarehouseTransfer $transfer): WarehouseTransfer
    {
        // Make the payload for creating a new warehouse transfer
        $payload = [
            'from_warehouse_id' => $transfer->from_warehouse_id,
            'to_warehouse_id' => $transfer->to_warehouse_id,
            'shipping_method_id' => $transfer->shipping_method_id,
            'products' => $transfer->warehouseTransferLines->map(function (WarehouseTransferLine $transferLine) {
                return ['id' => $transferLine->product_id, 'quantity' => $transferLine->quantity];
            })->toArray(),
        ];

        // Initiate the new transfer and return it
        return $this->initiateTransfer($payload);
    }
}
