<?php

namespace App\Managers;

use App\Models\FifoLayer;
use App\Models\SalesOrderLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Exception;
use Throwable;

class SalesCreditManager
{
    // For a given sales credit line, get the unit cost
    /**
     * @throws Exception
     * @throws Throwable
     */
    public function getUnitCost(SalesOrderLine $salesOrderLine, $qtyToCredit): ?float
    {
        if (! $salesOrderLine->product) {
            return null;
        }

        throw_if($qtyToCredit == 0, 'Quantity to credit is 0');
        $qtyToCreditTally = $qtyToCredit;
        $salesOrderLineQty = $salesOrderLine->quantity;

        // Get the layers of the sales order line associated to the sales credit line
        $salesOrderLineLayers = $salesOrderLine->salesOrderLineLayers;

        if ($salesOrderLineLayers->isEmpty()) {
            return $salesOrderLine->product->unit_cost;
        }

        $salesOrderLineLayersQuantity = $salesOrderLineLayers->sum('quantity');

        $backorderQueue = $salesOrderLine->backorderQueue;
        $backorderQueueQty = $backorderQueue?->unreleased_quantity ?? 0;

        throw_if($salesOrderLineQty != ($salesOrderLineLayersQuantity + $backorderQueueQty), 'Sales order line quantity does not match the sum of the layers and backorder queue');

        $totalUnitCostTally = 0;

        foreach ($salesOrderLineLayers as $salesOrderLineLayer) {
            /** @var FifoLayer $fifoLayer */
            $fifoLayer = $salesOrderLineLayer->layer;
            $qtyApplied = min($qtyToCreditTally, $salesOrderLineLayer->quantity);
            $totalUnitCostTally += $qtyApplied * $fifoLayer->avg_cost;
            $qtyToCreditTally -= $qtyApplied;
        }

        if ($backorderQueue) {
            if ($qtyToCreditTally > 0) {
                $salesOrderLineFinancialManager = new SalesOrderLineFinancialManager();
                $qtyApplied = min($qtyToCreditTally, $backorderQueueQty);
                $totalUnitCostTally += $qtyApplied * $salesOrderLineFinancialManager->getCogs($salesOrderLine);
            }
        }

        return $totalUnitCostTally / $qtyToCredit;
    }
}
