<?php

namespace App\Managers;

use App\Data\FinancialSummaryQueryData;
use App\Models\ProductListing;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\ProductRepository;
use App\Repositories\ReportingDailyFinancialRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class ReportManager
{
    public function __construct(
        protected readonly InventoryMovementRepository $inventoryMovementRepository,
        protected readonly FifoLayerRepository $fifoLayerRepository,
        protected readonly SalesOrderLineFinancialsRepository $lineFinancials,
        protected readonly ReportingDailyFinancialRepository $dailyFinancials,
        protected readonly ProductRepository $products,
    ) {
    }

    public function agingInventory(): EloquentCollection
    {
        return $this->fifoLayerRepository->getAgingInventory();
    }

    public function valuationSummary(): array
    {
        return Cache::remember('valuation-summary', 60, function () {
            return $this->inventoryMovementRepository
                ->getValuationSummaryByInventoryStatus();
        });
    }

    /**
     * @throws \Exception
     */
    public function valuationSummaryByProduct(): array
    {
        return [
            'products' => $this->inventoryMovementRepository
                ->getValuationSummaryByInventoryStatusAndProduct()
                ->toArray(),
            'totals' => $this->inventoryMovementRepository
                ->getValuationSummaryByInventoryStatus()
        ];
    }

    public function topProducts(string $metric, int $limit): EloquentCollection
    {
        return $this->lineFinancials->getTopProducts($metric, $limit);
    }

    /**
    * This method can be called dynamically by the getFinancialsBy method in the ReportingController class.
    */
    public function getFinancialsByBrand(FinancialSummaryQueryData $data): array
    {
        $filter = request('filter');
        if(isset($filter['integration'])) {
            return $this->getFinancials($this->lineFinancials->getFinancialsByBrand($data), ['Brand']);
        }
        return $this->getFinancials($this->dailyFinancials->getFinancialsByBrand($data), ['Brand']);
    }

    /**
     * This method can be called dynamically by the getFinancialsBy method in the ReportingController class.
     */
    public function getFinancialsBySupplier(FinancialSummaryQueryData $data): array
    {
        $filter = request('filter');
        if(isset($filter['integration'])) {
            return $this->getFinancials($this->lineFinancials->getFinancialsBySupplier($data), ['Supplier']);
        }
        return $this->getFinancials($this->dailyFinancials->getFinancialsBySupplier($data), ['Supplier']);
    }

    /**
     * This method can be called dynamically by the getFinancialsBy method in the ReportingController class.
     */
    public function getFinancialsBySku(FinancialSummaryQueryData $data): array
    {
        $filter = request('filter');
        if(isset($filter['integration'])) {
            return $this->getFinancials($this->lineFinancials->getFinancialsBySku($data), [
                'SKU',
                'Name',
                'Brand',
                'Supplier',
            ]);
        }
        return $this->getFinancials($this->dailyFinancials->getFinancialsBySku($data), [
            'SKU',
            'Name',
            'Brand',
            'Supplier',
        ]);
    }

    /**
     * This method can be called dynamically by the getFinancialsBy method in the ReportingController class.
     */
    public function getFinancialsByProductType(FinancialSummaryQueryData $data): array
    {
        return $this->getFinancials($this->lineFinancials->getFinancialsByProductType($data), ['Type']);
    }

    /**
     * This method can be called dynamically by the getFinancialsBy method in the ReportingController class.
     */
    public function getFinancialsBySalesChannel(FinancialSummaryQueryData $data): array
    {
        return $this->getFinancials($this->lineFinancials->getFinancialsBySalesChannel($data), ['Sales Channel']);
    }

    private function getUniqueIntervals(Collection $data): array
    {
        return $data->pluck('interval')->unique()->sort()->values()->all();
    }

    private function createPivotData(array $headers, Collection $data, array $intervals): array
    {
        return $data->groupBy('rowLabel')->map(function ($rows) use ($intervals, $headers) {
            $row = [
                $headers[0] => $rows->first()['rowLabel'],
            ];

            for ($i = 1; $i < count($headers); $i++) {
                $row[$headers[$i]] = $rows->first()[$headers[$i]];
            }

            foreach ($intervals as $interval) {
                $row[$interval] = $rows->where('interval', $interval)->sum('metric') ?: 0;
            }

            // Filter out non-numeric keys before summing
            $numericKeys = array_filter($row, function($key) {
                return is_numeric($key);
            }, ARRAY_FILTER_USE_KEY);

            $row['Total'] = array_sum($numericKeys);
            return $row;
        })->values()->all();
    }

    private function createTotalRow(string $header, array $pivotData, array $intervals): array
    {
        $totalRow = [$header => 'Total'];
        foreach ($intervals as $interval) {
            $totalRow[$interval] = array_sum(array_column($pivotData, $interval));
        }
        $totalRow['Total'] = array_sum(array_column($pivotData, 'Total'));
        return $totalRow;
    }

    private function createHeaders(array $headers, array $intervals): array
    {
        return array_merge($headers, $intervals, ['Total']);
    }

    private function getFinancials(Collection $data, array $headers): array
    {
        $intervals = $this->getUniqueIntervals($data);
        $pivotData = $this->createPivotData($headers, $data, $intervals);
        #$totalRow = $this->createTotalRow($header, $pivotData, $months);
        #$pivotData = array_merge($pivotData, [$totalRow]);
        $headers = $this->createHeaders($headers, $intervals);
        return [
            'headers' => $headers,
            'data' => $pivotData,
        ];
    }

    private function filterActiveListings(Collection $listings): Collection
    {
        return $listings->filter(function (ProductListing $listing) {
            $salesChannelProduct = $listing->document;
            return $salesChannelProduct->isActive();
        });
    }

    public function getSalesChannelCoverage(bool $byIntegration, bool $showNumberOfCoverages, array $ids): array
    {
        $data = $this->products->getSalesChannelCoverage();

        if ($byIntegration) {
            $groupBy = 'salesChannel.integrationInstance.integration.name';
        } else {
            $groupBy = 'salesChannel.integrationInstance.name';
        }

        $reportData = $data->map(function ($product) use ($groupBy, $byIntegration, $showNumberOfCoverages, $ids) {
            $coverage = $this->calculateSalesChannelCoverage($product->productListings, $groupBy, $byIntegration, $showNumberOfCoverages, $ids);

            if ($product->parentProducts->count() > 0) {
                $parentCoverage = $product->parentProducts->map(function ($parentProduct) use ($groupBy, $byIntegration, $showNumberOfCoverages, $ids) {
                    return $this->calculateSalesChannelCoverage($parentProduct->productListings, $groupBy, $byIntegration, $showNumberOfCoverages, $ids);
                });

                $parentCoverage = $parentCoverage->reduce(function ($carry, $item) use ($showNumberOfCoverages) {
                    if (is_null($carry)) {
                        $carry = new Collection();
                    }

                    foreach ($item as $key => $value) {
                        if ($carry->has($key)) {
                            $carry[$key] += $value;
                        } else {
                            $carry->put($key, $value);
                        }
                    }

                    if (!$showNumberOfCoverages) {
                        $carry = $carry->map(function ($value) {
                            return min(1, $value);
                        });
                    }

                    return $carry;
                });

                if ($coverage->count() == 0) {
                    $coverage = $parentCoverage;
                }
                else {
                    $coverage = $coverage->merge($parentCoverage)->mapWithKeys(function ($value, $key) use ($parentCoverage, $showNumberOfCoverages) {
                        $newValue = $value + ($parentCoverage[$key] ?? 0);
                        if (!$showNumberOfCoverages) {
                            $newValue = min(1, $newValue);
                        }
                        return [$key => $newValue];
                    });
                }
            }

            // Adjust group by name to replace . with _
            $coverage = $coverage->mapWithKeys(function ($value, $key) {
                return [str_replace('.', '_', $key) => $value];
            });

            return [
                'id' => $product->id,
                'sku' => $product->sku,
                'type' => $product->type,
                'stock' => $product->productInventory->where('warehouse_id', 0)->sum('inventory_available'),
                'valuation' => $product->activeFifoLayers->sum('actual_cost'),
                'supplier' => [
                    'id' => $product->defaultSupplierProduct?->supplier->id,
                    'name' => $product->defaultSupplierProduct?->supplier->name,
                ],
                'coverage' => $coverage->count() == 0 ? ['Total' => 0] : $coverage->merge(['Total' => $coverage->sum()])->toArray()
            ];
        });

        return $reportData->toArray();
    }

    private function calculateSalesChannelCoverage(Collection $listings, string $groupBy, bool $byIntegration, bool $showNumberOfCoverages, array $ids)
    {
        return $listings->groupBy($groupBy)->map(function ($listings) use ($byIntegration, $showNumberOfCoverages, $ids) {
            $listings = $listings->filter(function (ProductListing $listing) use ($byIntegration, $ids) {
                if ($byIntegration == 'integration') {
                    return in_array($listing->salesChannel->integrationInstance->integration->id, $ids);
                } else {
                    return in_array($listing->salesChannel->integrationInstance->id, $ids);
                }
            });
            $listings = $this->filterActiveListings($listings);
            return $showNumberOfCoverages ? $listings->count() : ($listings->count() ? 1 : 0);
        });
    }
}
