<?php

namespace App\Managers;

use App\Data\InitialInventoryData;
use App\Exceptions\UsedFifoLayerException;
use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\Setting;
use App\Repositories\ProductRepository;
use App\Services\Product\ReleaseBackorderQueues;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class ProductManager
{
    public function __construct(
        private readonly ProductRepository $products
    )
    {}

    /**
     * @throws Exception
     */
    public function evaluateTemplate(string|array $template, Product $product): null|string|float|array
    {
        // Replace placeholders first
        $template = preg_replace_callback('/{{(.*?)}}/', function ($matches) use ($product) {
            $variable = trim($matches[1]);
            $resolvedValue = $this->resolveFieldValue($variable, $product);

            // Check if the resolved value is an array
            if (is_array($resolvedValue)) {
                return json_encode($resolvedValue); // Convert array to JSON string for further processing
            }

            return $resolvedValue;
        }, $template);

        // Handle empty array
        if ($template == "[]") {
            return [];
        }

        // Decode JSON back to array if it was converted
        if ($decodedTemplate = json_decode($template, true)) {
            if (is_array($decodedTemplate)) {
                return $decodedTemplate;
            }
        }

        // Check if the template contains any mathematical expressions
        if ($this->isFormula($template)) {
            return $this->evaluateFormula($template);
        }

        return $template;
    }

    protected function isFormula($template): bool|int
    {
        // Check if the template is a mathematical formula
        return preg_match('/^\s*[-+]?\d+(\.\d+)?\s*[-+*\/]\s*[-+]?\d+(\.\d+)?(\s*[-+*\/]\s*[-+]?\d+(\.\d+)?)*\s*$/', $template);
    }

    /**
     * @throws Exception
     */
    protected function resolveFieldValue(string $path, Product $product): null|string|float|array
    {
        $pathParts = explode('.', $path);

        return match ($pathParts[0]) {
            'product' => $product->{$pathParts[1]},
            'product_brand' => $product->brand->name,
            'product_blemished' => $product->productBlemished?->{$pathParts[1]} ?? '',
            'product_images' => $product->images->sortBy('sort_order')->pluck('relative_url')->toArray(),
            'product_pricing' => $product->productPricingTiers
                ->where('name', $pathParts[1])->first()?->productPrices
                ->where('product_id', $product->id)->first()?->price,
            'product_attributes' => $product->productAttributes->where('name', $pathParts[1])->first()?->pivot->value,
            default => throw new Exception('Invalid path provided for dynamic field value resolution: ' . $path),
        };
    }

    /**
     * @throws Exception
     */
    protected function evaluateFormula($formula): float
    {
        // Remove spaces from the formula
        $formula = str_replace(' ', '', $formula);

        // Safely evaluate the formula
        try {
            // Use a safer evaluation method, avoiding eval()
            $result = 0;
            eval('$result = ' . $formula . ';');
            return round($result, 2);
        } catch (Exception $e) {
            throw new Exception("Failed to evaluate formula: {$formula}");
        }
    }
}