<?php

namespace App\Managers;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstance;
use App\Data\SalesOrderLineMappingData;
use App\DTO\ProductListingDto;
use App\DTO\ProductMappingDto;
use App\Exceptions\SalesChannelProductMappingException;
use App\Models\IntegrationInstance;
use App\Jobs\Shopify\ShopifyMapSalesOrderLines;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderLineItem;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Repositories\ProductListingRepository;
use App\Repositories\ProductRepository;
use App\Repositories\WarehouseRepository;
use App\Services\SalesOrder\SalesOrderManager;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Throwable;

/**
 * @deprecated see AbstractSalesChannelProductManager
 */
class ProductListingManager
{
    private ProductRepository $products;

    private ProductListingRepository $listings;

    public function __construct()
    {
        $this->products = app(ProductRepository::class);
        $this->listings = app(ProductListingRepository::class);
    }

    private function updateProductListingForSalesOrderLines(Collection $productListingCollection, Model $salesChannelProduct)
    {
        $salesChannelOrderClass = null;
        $salesChannelOrderLineClass = null;

        if (get_class($salesChannelProduct) == ShopifyProduct::class) {
            $salesChannelOrderClass = app(ShopifyOrder::class);
            $salesChannelOrderLineClass = app(ShopifyOrderLineItem::class);
        } else {
            throw new Exception('Missing table information', 1);
        }

        $salesOrderLines = SalesOrderLine::select(
            'sales_order_lines.id as id',
            'pl.product_id as product_id',
            'pl.id as product_listing_id'
        )
            ->joinRelationshipUsingAlias('salesOrder', 'so')
            ->join($salesChannelOrderLineClass->getTable().' as scl', 'sales_order_lines.sales_channel_line_id', '=', 'scl.'.$salesChannelOrderLineClass::getUniqueId())
            ->join($salesChannelProduct->getTable().' as scp', function ($join) use ($salesChannelProduct, $salesChannelOrderLineClass) {
                $join->on('scl.'.$salesChannelOrderLineClass->salesChannelProductSkuColumn(), 'scp.'.$salesChannelProduct::getUniqueField());
            })
            ->join((new ProductListing())->getTable().' as pl', function ($join) use ($salesChannelProduct) {
                $join->on('pl.sales_channel_listing_id', 'scp.'.$salesChannelProduct::getUniqueField())->where('pl.document_type', get_class($salesChannelProduct));
            })
            ->where('so.sales_channel_order_type', get_class($salesChannelOrderClass))
            ->whereIn('pl.id', $productListingCollection->pluck('id')->values())
            ->whereNull('sales_order_lines.product_listing_id')
            ->get();

        app(SalesOrderManager::class)->mapSalesOrderLinesToSalesChannelProducts(SalesOrderLineMappingData::collection($salesOrderLines));

        SalesOrderLine::whereIn('sales_order_id', $salesOrderLines->pluck('id')->values())
            ->each(function ($salesOrderLine) {
                app(SalesOrderManager::class)->mapSalesOrderLine($salesOrderLine);
            });
    }

    /**
     * @throws Throwable
     */
    public function mapListings(Collection $mappingData, Model $salesChannelProduct, IntegrationInstanceInterface $integrationInstance)
    {
        //Hydration code start here
        //Get sales channel products
        $salesChannelProducts = $salesChannelProduct::whereIn($salesChannelProduct::getUniqueField(), $mappingData->pluck('sales_channel_listing_id'))->get();

        //Get products
        $products = $this->products->getProductsFromSkus($mappingData->pluck('mapped_sku')->values()->all());

        //Get existing product listings
        $productListings = $this->listings->getFromSalesChannelIdFromUniqueSalesChannelProductIds(
            $integrationInstance->salesChannel->id,
            $mappingData->pluck('sales_channel_listing_id')->toArray()
        );

        $productMappingData = $mappingData->map(function (ProductMappingDto $mapping) use ($salesChannelProducts, $salesChannelProduct, $products, $productListings) {
            $mapping->salesChannelProduct = $salesChannelProducts->where($salesChannelProduct::getUniqueField(), $mapping->sales_channel_listing_id)->first();
            $mapping->product = $products->where('sku', $mapping->mapped_sku)->first();
            $mapping->product_listing_id = $productListings->where('sales_channel_listing_id', $mapping->sales_channel_listing_id)->first()?->id;

            return $mapping;
        });
        //Hydration code end here

        //Unmap start here
        $unmappedProductListings = $productMappingData
            ->whereNull('mapped_sku')
            ->whereNotNull('product_listing_id')
            ->pluck('product_listing_id');
        if ($unmappedProductListings->count()) {
            $this->listings->deleteProductListingsForIds($unmappedProductListings->toArray());
            $productMappingData = $productMappingData->reject(fn ($mapping) => $unmappedProductListings->contains($mapping->product_listing_id));
        }
        //Unmap end here

        $productListingCollection = ProductListingDto::collection($productMappingData->map(function (ProductMappingDto $mapping) use ($salesChannelProduct, $integrationInstance) {
            $salesChannelProduct = $mapping->salesChannelProduct;

            if (! $mapping->product) {
                throw new SalesChannelProductMappingException('Trying to map '.$mapping->sales_channel_listing_id.' to a non existing product '.$mapping->mapped_sku);
            }

            return ProductListingDto::from([
                'id' => $mapping->product_listing_id, // Either new mapping (if null) or remapping
                'document_id' => $salesChannelProduct->id,
                'document_type' => get_class($salesChannelProduct),
                'listing_sku' => $salesChannelProduct->{$salesChannelProduct::getSkuField()},
                'sales_channel_id' => $integrationInstance->salesChannel->id,
                'sales_channel_listing_id' => $salesChannelProduct->{$salesChannelProduct::getUniqueField()},
                'product_id' => $mapping->product->id,
            ]);
        }));

        $productListingCollection = $this->listings->saveWithRelations($productListingCollection);

        $this->updateProductListingForSalesOrderLines($productListingCollection, $salesChannelProduct);

        $productListingCollection->each(function ($productListing) use ($integrationInstance) {
            $productListing = ProductListing::find($productListing['id']);
            dispatch_sync(new ShopifyMapSalesOrderLines($integrationInstance, $productListing, 'new'));
        });

    }

    public function cacheInventoryQuantity(IntegrationInstance|AbstractSalesChannelIntegrationInstance $integrationInstance, array $productIdsToCache = []): void
    {
        $productIdsToCache = !empty($productIdsToCache) ? $productIdsToCache : $this->products->getProductIdsNeedingListingCacheUpdate($integrationInstance);
        // WIP
    }
}
