<?php

namespace App\Managers;

use App\Data\CreateInventoryMovementData;
use App\Data\FifoLayerQuantityData;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Repositories\InventoryMovementRepository;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class InventoryMovementManager
{
    public function __construct(
        private readonly InventoryMovementRepository $movements
    )
    {
    }

    /**
     * @throws Throwable
     */
    public function createSalesOrderReservationMovementsForFifo(SalesOrderLine $salesOrderLine, FifoLayerQuantityData $fifoLayerQuantityData): Collection
    {
        return $this->createSalesOrderReservationMovements(
            $salesOrderLine,
            $fifoLayerQuantityData->quantity,
            $fifoLayerQuantityData->fifoLayer->id,
            FifoLayer::class
        );
    }

    /**
     * @throws Throwable
     */
    public function createSalesOrderReservationMovementsForBackorder(SalesOrderLine $salesOrderLine, BackorderQueue $backorder): Collection
    {
        return $this->createSalesOrderReservationMovements(
            $salesOrderLine,
            $backorder->backordered_quantity,
            $backorder->id,
            BackorderQueue::class
        );
    }

    /**
     * @throws Throwable
     */
    private function createSalesOrderReservationMovements(
        SalesOrderLine $salesOrderLine,
        int $quantity,
        int $layerId,
        string $layerType
    ): Collection {
        return DB::transaction(function () use ($salesOrderLine, $quantity, $layerId, $layerType) {
            $activeMovement = CreateInventoryMovementData::from([
                'product_id' => $salesOrderLine->product_id,
                'quantity' => -abs($quantity),
                'type' => InventoryMovement::TYPE_SALE,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'warehouse_id' => $salesOrderLine->warehouse_id,
                'link_id' => $salesOrderLine->id,
                'link_type' => SalesOrderLine::class,
                'layer_id' => $layerId,
                'layer_type' => $layerType,
                'reference' => $salesOrderLine->salesOrder->sales_order_number,
            ]);

            $reservationMovement = clone $activeMovement;
            $reservationMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
            $reservationMovement->quantity = abs($reservationMovement->quantity);

            if ($layerType === FifoLayer::class) {
                SalesOrderLineLayer::create([
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => abs($activeMovement->quantity),
                    'layer_id' => $layerId,
                    'layer_type' => FifoLayer::class,
                ]);
            }

            return collect([
                InventoryMovement::create($activeMovement->toArray()),
                InventoryMovement::create($reservationMovement->toArray()),
            ]);
        });
    }

    public function createFulfillmentMovement(SalesOrderFulfillmentLine $salesOrderFulfillmentLine)
    {
        $reservationMovement = $this->movements->getCorrespondingReservationMovementForFulfillmentLineMovement($salesOrderFulfillmentLine);

        return InventoryMovement::create(CreateInventoryMovementData::from([
            'product_id' => $salesOrderFulfillmentLine->salesOrderLine->product_id,
            'quantity' => -abs($salesOrderFulfillmentLine->quantity),
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'warehouse_id' => $salesOrderFulfillmentLine->salesOrderLine->warehouse_id,
            'link_id' => $salesOrderFulfillmentLine->id,
            'link_type' => SalesOrderFulfillmentLine::class,
            'layer_id' => $reservationMovement->layer_id,
            'layer_type' => FifoLayer::class,
            'reference' => $salesOrderFulfillmentLine->salesOrderFulfillment->fulfillment_number,
        ])->toArray());
    }
}
