<?php

namespace App\Managers;

use App\Actions\InventoryHealth\BackorderFromFifoLayer;
use App\Actions\InventoryHealth\FixNonInventoryProductWithInventoryMovements;
use App\Actions\InventoryHealth\FixOversubscribedFifoLayer;
use App\Actions\InventoryHealth\FixSalesOrderLineLayerCache;
use App\Actions\InventoryHealth\UpdateSalesOrderLineLayerCache;
use App\Actions\InventoryManagement\CreateActiveInventoryMovementFromReserved;
use App\Actions\InventoryManagement\CreateReservedInventoryMovementFromActive;
use App\Actions\InventoryManagement\CreateSalesOrderReservation;
use App\Actions\InventoryManagement\DeleteSalesOrderReservation;
use App\Console\Commands\Inventory\Health\Fix\FixOrphanSalesOrderLineReservationMovementsCommandHelper;
use App\Data\CreateStockTakeData;
use App\Data\StockTakeItemData;
use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\FixOrphanSalesOrderLineReservationMovementUsingReference;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Services\StockTake\OpenStockTakeException;
use App\Services\StockTake\StockTakeManager;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class InventoryHealthManager
{
    public function __construct(
        private readonly CreateSalesOrderReservation $createSalesOrderReservation,
        private readonly DeleteSalesOrderReservation $deleteSalesOrderReservation,
        private readonly CreateActiveInventoryMovementFromReserved $createActiveInventoryMovementFromReserved,
        private readonly CreateReservedInventoryMovementFromActive $createReservedInventoryMovementFromActive,
        private readonly FixOversubscribedFifoLayer $fixOversubscribedFifoLayer,
        private readonly InventoryMovementManager $inventoryMovementManager,
        private readonly BackorderFromFifoLayer $backorderFromFifoLayer,
        private readonly FixSalesOrderLineLayerCache $fixSalesOrderLineLayerCache,
        private readonly UpdateSalesOrderLineLayerCache $updateSalesOrderLineLayerCache,
        private readonly StockTakeManager $stockTakeManager,
        private readonly FixOrphanSalesOrderLineReservationMovementUsingReference $fixOrphanSalesOrderLineReservationMovementUsingReference,
        private readonly FixNonInventoryProductWithInventoryMovements $fixNonInventoryProductWithInventoryMovements,
    ) {}

    /**
     * @throws UnableToAllocateToFifoLayersException
     * @throws Throwable
     */
    public function createSalesOrderReservation(SalesOrderLine $salesOrderLine): Collection {
        return ($this->createSalesOrderReservation)($salesOrderLine);
    }

    /**
     * @throws Throwable
     */
    public function deleteSalesOrderReservation(SalesOrderLine $salesOrderLine): Collection {
        return ($this->deleteSalesOrderReservation)($salesOrderLine);
    }

    /**
     * @throws OpenStockTakeException
     * @throws OversubscribedFifoLayerException
     * @throws InventoryMovementTypeException
     * @throws UnableToAllocateToFifoLayersException
     */
    public function createActiveInventoryMovementFromReserved(InventoryMovement $reservedInventoryMovement): Collection {
        return ($this->createActiveInventoryMovementFromReserved)($reservedInventoryMovement);
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     */
    public function createReservedInventoryMovementFromActive(InventoryMovement $activeInventoryMovement): Collection {
        return ($this->createReservedInventoryMovementFromActive)($activeInventoryMovement);
    }

    public function createFulfillmentInventoryMovementsForSalesOrderLine(SalesOrderLine $salesOrderLine): Collection {
        return $salesOrderLine->salesOrderFulfillmentLines->map(function (SalesOrderFulfillmentLine $salesOrderFulfillmentLine) use ($salesOrderLine) {
            if ($salesOrderFulfillmentLine->inventoryMovements->isNotEmpty()) {
                if ($salesOrderFulfillmentLine->inventoryMovements->sum('quantity') !== $salesOrderLine->inventoryMovements->sum('quantity')) {
                    throw new Exception('Fulfillment line inventory movements exist, but do not match sales order line inventory movements');
                }
                return false;
            }
            return $this->inventoryMovementManager->createFulfillmentMovement($salesOrderFulfillmentLine);
        })->filter();
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws UnableToAllocateToFifoLayersException
     * @throws InsufficientSalesOrderLineQuantityException
     */
    public function fixOverSubscribedFifoLayer(FifoLayer $fifoLayer): int {
        return ($this->fixOversubscribedFifoLayer)($fifoLayer);
    }

    public function backorderFromFifoLayer(FifoLayer $fifoLayer, int $quantityToBackorder): int {
        return ($this->backorderFromFifoLayer)($fifoLayer, $quantityToBackorder);
    }

    /**
     * @throws Exception
     */
    public function fixSalesOrderLineLayerCache(SalesOrderLine $salesOrderLine): void {
        ($this->fixSalesOrderLineLayerCache)($salesOrderLine);
    }

    /**
     * @throws Exception
     */
    public function updateSalesOrderLineLayerCache(
        SalesOrderLine $salesOrderLine,
        FifoLayer $fifoLayer,
        int $qtyChange
    ): void {
        ($this->updateSalesOrderLineLayerCache)($salesOrderLine, $fifoLayer, $qtyChange);
    }

    public function createStockTakesForOverages(Collection $fifoLayersToZeroOut, string $notes): Collection
    {
        $createdStockTakes = collect();
        $groupedByWarehouse = $fifoLayersToZeroOut->groupBy('warehouse_id');
        $groupedByWarehouse->each(/**
         * @throws Throwable
         */ function (Collection $fifoLayers, int $warehouseId) use ($notes, $createdStockTakes) {
            $stockTakeLines = collect();
            $productIds = $fifoLayers->pluck('product_id');
            $productIds->each(function (int $product_id) use ($stockTakeLines) {
                $stockTakeLines->add(StockTakeItemData::from([
                    'product_id' => $product_id,
                    'qty_counted' => 0,
                ]));
            });
            $this->createAndFinalizeInventoryFixStockTake($warehouseId, $stockTakeLines, $notes, $createdStockTakes);
        });
        return $createdStockTakes;
    }

    public function createStockTakeForProductWarehouseQuantities(Collection $productWarehouseQuantities, string $notes, bool $asAdjustment = false): Collection
    {
        $createdStockTakes = collect();
        $groupedByWarehouse = $productWarehouseQuantities->groupBy('warehouse_id');
        $groupedByWarehouse->each(/**
         * @throws Throwable
         */ function (Collection $productWarehouseQuantities, int $warehouseId) use ($notes, $createdStockTakes, $asAdjustment) {
            $stockTakeLines = collect();
            $productQuantities = $productWarehouseQuantities->groupBy('product_id')->map(fn($group) => $group->sum('quantity'));
            $productQuantities->each(function (int $quantity, int $product_id) use ($stockTakeLines, $asAdjustment) {
                $data = [
                    'product_id' => $product_id,
                ];
                if ($asAdjustment) {
                    $data['qty_change'] = $quantity;
                } else {
                    $data['qty_counted'] = $quantity;
                }
                $stockTakeLines->add(StockTakeItemData::from($data));
            });
            $this->createAndFinalizeInventoryFixStockTake($warehouseId, $stockTakeLines, $notes, $createdStockTakes);
        });
        return $createdStockTakes;
    }

    /**
     * @throws Throwable
     */
    private function createAndFinalizeInventoryFixStockTake(
        int $warehouseId,
        Collection $stockTakeLines,
        string $notes,
        Collection $createdStockTakes,
    ): void {
        $stockTakeData = CreateStockTakeData::from([
            'warehouse_id' => $warehouseId,
            'items' => $stockTakeLines,
            'notes' => $notes,
        ]);
        $createdStockTake = $this->stockTakeManager->createStockTake($stockTakeData);
        $createdStockTake = $this->stockTakeManager->initiateCount($createdStockTake, isIntegrityFix: true);
        $createdStockTake = $this->stockTakeManager->finalizeStockTake($createdStockTake, autoApplyStock: false);
        $createdStockTake->refresh();
        $warehouse = $createdStockTake->warehouse;
        $createdStockTakes->add($createdStockTake);
        customlog('inventory-fixes', "Stock take created (ID: $createdStockTake->id) for $warehouse->name", days: null);
    }

    /**
     * @throws Throwable
     */
    public function fixOrphanSalesOrderLineReservationMovementUsingReference(InventoryMovement $inventoryMovement): ?SalesOrderLine {
        return ($this->fixOrphanSalesOrderLineReservationMovementUsingReference)($inventoryMovement);
    }

    public function fixNonInventoryProductWithInventoryMovements(Product $product): void {
        ($this->fixNonInventoryProductWithInventoryMovements)($product);
    }
}
