<?php

namespace App\Managers;

use App\Data\CreateInventoryAdjustmentData;
use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Services\InventoryManagement\InventoryManager;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Throwable;

class InventoryAdjustmentManager
{
    /**
     * @throws Throwable
     */
    public function createAdjustment(CreateInventoryAdjustmentData $adjustmentData): ?InventoryAdjustment
    {
        return match ($adjustmentData->adjustment_type) {
            InventoryAdjustment::TYPE_INCREASE => $this->increaseInventory($adjustmentData),
            InventoryAdjustment::TYPE_DECREASE => $this->decreaseInventory($adjustmentData),
            InventoryAdjustment::TYPE_SET => $this->setInventory($adjustmentData),
            default => throw new InvalidArgumentException('Invalid adjustment type. valid values:'.implode(',',
                    InventoryAdjustment::ADJUSTMENT_TYPES)),
        };
    }

    /**
     * @throws Throwable
     */
    private function increaseInventory(CreateInventoryAdjustmentData $adjustmentData): InventoryAdjustment
    {
        $inventoryAdjustment = $this->createInventoryAdjustment($adjustmentData);
        return DB::transaction(function () use ($inventoryAdjustment, $adjustmentData) {
            InventoryManager::with(
                $inventoryAdjustment->warehouse_id,
                $inventoryAdjustment->product
            )->addToStock(
                quantity: abs($inventoryAdjustment->quantity),
                event: $inventoryAdjustment,
                unitCost: $adjustmentData->unit_cost
            );

            return $inventoryAdjustment;
        });
    }

    /**
     * @throws Throwable
     */
    private function decreaseInventory(CreateInventoryAdjustmentData $adjustmentData): InventoryAdjustment
    {
        $inventoryAdjustment = $this->createInventoryAdjustment($adjustmentData);
        return DB::transaction(function () use ($inventoryAdjustment) {

            InventoryManager::with(
                $inventoryAdjustment->warehouse_id,
                $inventoryAdjustment->product
            )->takeFromStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);

            return $inventoryAdjustment;
        });
    }

    /**
     * @throws Throwable
     */
    private function setInventory(CreateInventoryAdjustmentData $adjustmentData): ?InventoryAdjustment
    {
        $product = Product::with(['totalQuantity'])->findOrFail($adjustmentData->product_id);

        $warehouseInventoryCount = $product->inventoryMovements()
            ->where('warehouse_id', $adjustmentData->warehouse_id)
            ->sum('quantity');

        $diffQuantity = $adjustmentData->quantity - $warehouseInventoryCount;

        $adjustmentData->quantity = $diffQuantity;

        if ($diffQuantity != 0) {
            if ($diffQuantity > 0) {
                $adjustmentData->adjustment_type = InventoryAdjustment::TYPE_INCREASE;
                    return $this->increaseInventory($adjustmentData);
            } else {
                $adjustmentData->adjustment_type = InventoryAdjustment::TYPE_DECREASE;
                return $this->decreaseInventory($adjustmentData);
            }
        }

        return null;
    }

    private function getQuantity(CreateInventoryAdjustmentData $adjustmentData): float
    {
        $adjustmentType = $adjustmentData->adjustment_type;
        $quantity       = $adjustmentData->quantity;

        return match ($adjustmentType) {
            InventoryAdjustment::TYPE_INCREASE, InventoryAdjustment::TYPE_SET => abs($quantity),
            InventoryAdjustment::TYPE_DECREASE => -abs($quantity),
            default => $quantity,
        };
    }

    private function createInventoryAdjustment(CreateInventoryAdjustmentData $adjustmentData): InventoryAdjustment
    {
        $adjustmentData->quantity = $this->getQuantity($adjustmentData);
        return InventoryAdjustment::create($adjustmentData->toArray());
    }


}