<?php

namespace App\Managers;

use App\Data\CreateFinancialAlertData;
use App\Data\FinancialAlertSummaryNotificationData;
use App\Enums\FinancialAlertNotificationConditionEnum;
use App\Enums\FinancialAlertTypeEnum;
use App\Helpers;
use App\Models\FinancialAlert;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Notifications\FinancialAlertEmailNotification;
use App\Notifications\FinancialAlertSlackNotification;
use App\Repositories\FinancialAlertRepository;
use App\Repositories\SettingRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Notification;

class FinancialAlertManager
{
    public function __construct(
        private readonly FinancialAlertRepository $alerts,
        private readonly SettingRepository $settings,
    ) {
    }

    public function createFinancialAlerts(?Collection $salesOrderLines = null): void
    {
        // Repository method
        $alertedLines = $this->alerts->getSalesOrderLinesWithAlerts($salesOrderLines);

        $financialAlertsCollection = collect();

        $alertedLines->each(function (SalesOrderLine $line) use ($financialAlertsCollection) {
            $financialAlertsCollection->push($this->buildFinancialAlertData($line));
        });

        $this->alerts->save($financialAlertsCollection, FinancialAlert::class);
        $this->alerts->deleteResolvedAlerts();

        $this->sendNotification();

        $this->settings->set(Setting::KEY_FINANCIAL_ALERTS_LAST_NOTIFICATION_DATE, now()->format('Y-m-d H:i:s'));
    }

    public function buildFinancialAlertData(SalesOrderLine $line): CreateFinancialAlertData
    {
        return CreateFinancialAlertData::from([
            'sales_order_line_financial_id' => $line->salesOrderLineFinancial->id,
            'alert_type' => $line->salesOrderLineFinancial->profit < 0 ? FinancialAlertTypeEnum::LOSS : FinancialAlertTypeEnum::LOW_MARGIN,
        ]);
    }

    public function sendImmediateAlertNotification(FinancialAlert $financialAlert): void
    {
        $notificationType = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE);

        if (!$notificationType) {
            return;
        }

        $destination = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION);

        if (!$destination) {
            return;
        }

        if ($notificationType == Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_EMAIL) {
            $emails = explode(',', $destination);

            if (count($emails)) {
                Notification::route('mail', $emails)->notify(new FinancialAlertEmailNotification(
                    financialAlert: $financialAlert
                ));
            }
        } else if ($notificationType == Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_SLACK) {
            Notification::route('slack', $destination)->notify(new FinancialAlertSlackNotification(
                financialAlert: $financialAlert
            ));
        }
    }

    public function sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum $condition): void {
        $notificationType = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE);
        $notificationConditions = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS);

        if (!$notificationType) {
            return;
        }

        if (!in_array($condition->value, json_decode($notificationConditions, true))) {
            return;
        }

        $destination = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION);

        if (!$destination) {
            return;
        }

        $data = $this->getSummaryData($condition);

        if ($notificationType == Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_EMAIL) {
            $emails = explode(',', $destination);

            if (count($emails)) {
                Notification::route('mail', $emails)->notify(new FinancialAlertEmailNotification(
                    data: $data,
                    condition: $condition
                ));
            }
        } else if ($notificationType == Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_SLACK) {
            Notification::route('slack', $destination)->notify(new FinancialAlertSlackNotification(
                data: $data,
                condition: $condition
            ));
        }
    }

    public function getSummaryData(FinancialAlertNotificationConditionEnum $condition): FinancialAlertSummaryNotificationData
    {
        $date = $condition == FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY ? Carbon::now()->subWeek()->startOfDay() : Carbon::now()->startOfDay();

        $countByType = FinancialAlert::selectRaw('alert_type, COUNT(*) as alert_count')
            ->where('created_at', '>=', $date)
            ->archived()
            ->groupBy('alert_type')
            ->get();

        $countByTypeAllTime = FinancialAlert::selectRaw('alert_type, COUNT(*) as alert_count')
            ->archived()
            ->groupBy('alert_type')
            ->get();

        $affectedOrdersQuery = FinancialAlert::leftJoin('sales_order_line_financials', 'financial_alerts.sales_order_line_financial_id',  '=', 'sales_order_line_financials.id')
            ->leftJoin('sales_order_lines', 'sales_order_line_financials.sales_order_line_id', '=', 'sales_order_lines.id')
            ->selectRaw('count(DISTINCT sales_order_lines.sales_order_id) as order_count')
            ->whereNull('financial_alerts.archived_at');

        $affectedOrders = $affectedOrdersQuery->clone()->where('financial_alerts.created_at', '>=', $date)->first();
        $affectedOrdersAllTime = $affectedOrdersQuery->clone()->first();

        return FinancialAlertSummaryNotificationData::from([
            'new_alert_count' => $countByType->sum('alert_count'),
            'new_loss_count' => $countByType->where('alert_type', FinancialAlertTypeEnum::LOSS)->sum('alert_count'),
            'new_loss_margin_count' => $countByType->where('alert_type', FinancialAlertTypeEnum::LOW_MARGIN)->sum('alert_count'),
            'new_order_count' => $affectedOrders->order_count,

            'total_alert_count' => $countByTypeAllTime->sum('alert_count'),
            'total_loss_count' => $countByTypeAllTime->where('alert_type', FinancialAlertTypeEnum::LOSS)->sum('alert_count'),
            'total_loss_margin_count' => $countByTypeAllTime->where('alert_type', FinancialAlertTypeEnum::LOW_MARGIN)->sum('alert_count'),
            'total_order_count' => $affectedOrdersAllTime->order_count
        ]);
    }

    public function sendNotification(): void
    {
        $notificationConditions = Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS);

        if (!$notificationConditions || !in_array(FinancialAlertNotificationConditionEnum::IMMEDIATELY->value, json_decode($notificationConditions))) {
            return;
        }
        $lastNotificationDate = $this->settings->get(Setting::KEY_FINANCIAL_ALERTS_LAST_NOTIFICATION_DATE);

        if ($lastNotificationDate) {

            $financialAlerts = $this->alerts->getLastFinancialAlertsForNotification($lastNotificationDate);

            if ($financialAlerts->count() <= 5) {
                $financialAlerts->each(function (FinancialAlert $alert) {
                    $this->sendImmediateAlertNotification($alert);
                });
            } else {
                $this->sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum::DAILY_SUMMARY);
            }
        }
    }
}