<?php

namespace Tests\App\Lib\DataList;

use App\Lib\DataList\DataListFactory;
use Tests\InvokePrivateMethodTrait;
use Tests\TestCase;

class DataListFactoryTest extends TestCase
{
    use InvokePrivateMethodTrait;

    private const DATA_LIST_PATH = __DIR__.'/files/';

    /**
     * @dataProvider fileNamesDataProvider
     */
    public function testLoadSimple(string $fileName): void
    {
        $dataList = DataListFactory::load($fileName);

        $columnHeadings = $dataList->getColumnHeadings();

        $this->assertEquals(
            ['Name', 'Age'],
            $columnHeadings
        );

        $rowsGenerator = $dataList->getRows();

        $this->assertEquals(
            ['John', '34'],
            array_values($rowsGenerator->current())
        );

        $rowsGenerator->next();
        $this->assertEquals(
            ['Paul', '17'],
            array_values($rowsGenerator->current())
        );

        $rowsGenerator->next();
        $this->assertEquals(
            ['Greg', '28'],
            array_values($rowsGenerator->current())
        );

        $rowsGenerator->next();
        $this->assertNull($rowsGenerator->current());
    }

    public function testLoadComplex(): void
    {
        $dataList = DataListFactory::load(self::DATA_LIST_PATH.'complex/utf-8-unix-complex.csv');

        $columnHeadings = $dataList->getColumnHeadings();

        $this->assertEquals(
            ['My column heading'],
            $columnHeadings
        );

        $rowsGenerator = $dataList->getRows();

        $this->assertMatchesRegularExpression(
            '~This is a\Rsentence with\Rnew lines~',
            array_values($rowsGenerator->current())[0]
        );

        $rowsGenerator->next();
        $this->assertEquals(
            ['This has a " double quote'],
            array_values($rowsGenerator->current())
        );

        $rowsGenerator->next();
        $this->assertEquals(
            ['This has a \' single quote'],
            array_values($rowsGenerator->current())
        );

        $rowsGenerator->next();
        $this->assertNull($rowsGenerator->current());
    }

    /**
     * @dataProvider fileNamesUnusualHeadingsDataProvider
     */
    public function testLoadUnusualHeadings($fileName): void
    {
        $dataList = DataListFactory::load($fileName);

        $columnHeadings = $dataList->getColumnHeadings();

        $expectedHeadings = [
            'sku', 'id', 'image_url', 'name', 'brand', 'price.Retail.value',
            'price.Retail.currency', 'price.Retail Test 2.value', 'price.Retail Test 2.currency',
            'price.Amazon US MFN.value', 'price.Amazon US MFN.currency', 'price.Amazon US FBA.value',
            'price.Amazon US FBA.currency', 'price.Amazon US Competitive Pricee.value',
            'price.Amazon US Competitive Pricee.currency', 'average_cost', 'average_cost_currency',
            'default_supplier_name', 'supplier_pricing.Wholesale.value',
            'supplier_pricing.Wholesale.currency', 'inventory_stock_value',
        ];

        $this->assertEquals(
            $expectedHeadings,
            $columnHeadings
        );
    }

    public function testGuessEncoding(): void
    {
        $encoding = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'guessEncoding',
            [self::DATA_LIST_PATH.'simple/utf-8.csv']
        );
        $this->assertEquals('UTF-8', $encoding);

        $encoding = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'guessEncoding',
            [self::DATA_LIST_PATH.'simple/utf-16.csv']
        );
        $this->assertEquals('UTF-16', $encoding);
    }

    public function testGuessFileType(): void
    {
        $fileType = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'guessFileType',
            [self::DATA_LIST_PATH.'simple/utf-8.csv']
        );
        $this->assertEquals('csv', $fileType);

        $fileType = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'guessFileType',
            [self::DATA_LIST_PATH.'simple/google-docs.xlsx']
        );
        $this->assertEquals('xlsx', $fileType);

        $fileType = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'guessFileType',
            [self::DATA_LIST_PATH.'simple/libre-office.ods']
        );
        $this->assertEquals('ods', $fileType);
    }

    public function testConvertEncoding(): void
    {
        $outputFileName = tempnam(sys_get_temp_dir(), 'DataListFactoryTest').'.csv';

        $fileType = $this->invokePrivateStaticMethod(
            DataListFactory::class,
            'convertEncoding',
            [
                self::DATA_LIST_PATH.'simple/utf-16.csv',
                'UTF-16',
                $outputFileName,
                'UTF-8',
            ]
        );
        $this->assertFileExists($outputFileName);

        $encoding = mb_detect_encoding(
            file_get_contents($outputFileName),
            ['UTF-8', 'UTF-16']
        );

        $this->assertEquals('UTF-8', $encoding);
    }

    /**
     * @dataProvider rowCountDataProvider
     */
    public function testGetRowCount(string $fileName, int $expectedRowCount): void
    {
        $dataList = DataListFactory::load($fileName);

        $actualRowCount = $dataList->getRowCount();

        $this->assertEquals($expectedRowCount, $actualRowCount);
    }

    public function rowCountDataProvider(): array
    {
        return [
            [self::DATA_LIST_PATH.'complex/utf-8-blank-lines-ending.csv', 1],
            [self::DATA_LIST_PATH.'simple/utf-16.csv', 3],
            [self::DATA_LIST_PATH.'simple/google-docs.xlsx', 3],
        ];
    }

    public function test_trimmed(): void
    {
        $dataList = DataListFactory::load(self::DATA_LIST_PATH.'complex/utf-8-untrimmed.csv');

        $this->assertEquals(
            ['this heading is untrimmed', 'this heading is also untrimmed'],
            $dataList->getColumnHeadings()
        );

        $this->assertEquals(
            ['this value is untrimmed', 'this value is also untrimmed'],
            array_values($dataList->getRows()->current())
        );
    }

    public function fileNamesUnusualHeadingsDataProvider(): array
    {
        return [
            [self::DATA_LIST_PATH.'complex/utf-16-unusual-headings.csv'],
        ];
    }

    public function fileNamesDataProvider(): array
    {
        return [
            'TSV' => [self::DATA_LIST_PATH.'simple/utf-8.tsv'],
            'GoogleDocsXlsx' => [self::DATA_LIST_PATH.'simple/google-docs.xlsx'],
            'OpenOfficeXml' => [self::DATA_LIST_PATH.'simple/office-open-xml.xlsx'],
            'UTF-8_CSV' => [self::DATA_LIST_PATH.'simple/utf-8.csv'],
            'UTF-16_CSV' => [self::DATA_LIST_PATH.'simple/utf-16.csv'],
        ];
    }

    protected function getTimeInFractionalSeconds(): float
    {
        $milliSeconds = (new \DateTime())->format('v');
        $seconds = (new \DateTime())->format('U');

        return $seconds + ($milliSeconds / 1000);
    }
}
