<?php

namespace App\Jobs;

use App\Models\Currency;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\SalesOrderLine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;

class UpdateDefaultCurrencyJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * the currency that you want to convert the values from it (Previous default currency).
     *
     * @var int
     */
    protected $fromCurrencyId;

    /**
     * @var Currency
     */
    protected $fromCurrency;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(int $fromCurrencyId)
    {
        $this->fromCurrencyId = $fromCurrencyId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // fetch currency to get its conversion rate to the default currency
        $this->fromCurrency = Currency::with([])->findOrFail($this->fromCurrencyId);

        // update FifoLayers
        $this->updateFifoLayers();

        // update average cost of products
        $this->updateProductsAverageCost();

        // update unit_cost of sales order lines
        $this->updateSalesOrderLinesUnitCost();
    }

    /**
     * Update Currency columns for FifoLayers.
     */
    private function updateFifoLayers()
    {
        FifoLayer::query()->update(['total_cost' => DB::raw("`total_cost` / {$this->fromCurrency->conversion}")]);
    }

    /**
     * Update Currency columns for Products' Average Cost.
     */
    private function updateProductsAverageCost()
    {
        $productAvgCostFromFifoLayersQuery = 'SELECT SUM((`total_cost` / `original_quantity`) * `available_quantity`) / SUM(`available_quantity`) FROM `fifo_layers` WHERE `fifo_layers`.`available_quantity` > 0 AND `fifo_layers`.`product_id` = `products`.`id`';

        // only update products that have an average_cost, means that have fifo layers
        Product::query()->where('average_cost', '>', 0)->update(['average_cost' => DB::raw("({$productAvgCostFromFifoLayersQuery})")]);
    }

    /**
     * Update Currency columns for Sales Orders' Unit Cost.
     */
    private function updateSalesOrderLinesUnitCost()
    {
        SalesOrderLine::query()->update(['unit_cost' => DB::raw("`unit_cost` / {$this->fromCurrency->conversion}")]);
    }
}
