<?php

namespace App\Jobs;

use App\Data\NoteData;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\UserRepository;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Notification;
use App\Notifications\UncancellableVeracoreOrderNotification;
use Modules\Veracore\Entities\VeracoreOrder;
use Spatie\LaravelData\Optional;

class UncancellableVeracoreOrderJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private SalesOrderRepository $salesOrders;
    private UserRepository $users;

    // Define the statuses that indicate a problem
    private array $problematicStatuses = [
        'Shipped',
        'Complete',
        'Pending',
        'Picked',
    ];

    /**
     * Create a new job instance.
     */
    public function __construct(
        public readonly SalesOrderFulfillment $fulfillment,
        public readonly array $currentOrder,
        public readonly VeracoreOrder $veracoreOrder,
        public readonly ?int $userId,
    )
    {
        $this->salesOrders = app(SalesOrderRepository::class);
        $this->users = app(UserRepository::class);
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $note = $this->getNoteContent();
        $this->addNoteToSalesOrder($note);
        $this->sendNotification($note);
    }

    /**
     * Add a note to the sales order.
     */
    private function addNoteToSalesOrder(string $note): void
    {
        $this->salesOrders->addNote(
            salesOrder: $this->fulfillment->salesOrder,
            data: NoteData::from([
                'link_id' => $this->fulfillment->salesOrder->id,
                'link_type' => SalesOrder::class,
                'user_id' => $this->userId ?? Optional::create(),
                'note' => $note
            ])
        );
    }

    /**
     * Generate the content of the note.
     */
    public function getNoteContent(): string
    {
        $problematicStatuses = $this->getProblematicStatuses();
        $suffix = count($problematicStatuses) === 1
            ? "$problematicStatuses[0]."
            : "one of " . implode(', ', $problematicStatuses) . ".";
        return "We attempted to cancel the fulfillment at Veracore, but it failed due to the order status being $suffix";
    }

    /**
     * Get the problematic statuses from the Veracore order.
     */
    private function getProblematicStatuses(): array
    {
        $orderStatuses = $this->currentOrder['OrdHead']['Status'];
        $statuses = [];
        foreach(array_keys($orderStatuses) as $status){
            if(in_array($status, $this->problematicStatuses) && $orderStatuses[$status] == 'true'){
                $statuses[] = $status;
            }
        }
        return $statuses;
    }

    /**
     * Send a notification to the admins.
     */
    private function sendNotification(string $note): void
    {
        Notification::send(
            notifiables: $this->users->getAdminUsers(),
            notification: new UncancellableVeracoreOrderNotification(
                salesOrder: $this->fulfillment->salesOrder,
                note: $note
            )
        );

        $this->veracoreOrder->update([
            'notified_about_cancellation' => true
        ]);
    }
}