<?php

namespace App\Jobs;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\SalesOrderFulfillment;
use App\Notifications\SalesOrderFulfilledNotification;
use App\Services\SalesOrder\Fulfillments\Dispatchers\ShipmentDispatcherFactory;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Notification;
use Throwable;

class SubmitFulfillmentToThirdPartiesJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public SalesOrderFulfillment $fulfillment
    )
    {
        $this->onQueue('fulfillments');
    }

    /**
     * @return void
     * @throws BindingResolutionException
     * @throws SalesOrderFulfillmentDispatchException
     * @throws Throwable
     */
    public function handle(): void
    {
        if(!$this->fulfillment->is_manual){
            $this->submitFulfillmentToShippingProvider();
        } else if($this->fulfillment->fulfilled()){
            SubmitTrackingInfo::factory($this->fulfillment)?->submit();
            // Send email notifications if needed
            if ($this->fulfillment->salesOrder->salesChannel->emailsCustomers()) {
                Notification::send(
                    notifiables: [$this->fulfillment->salesOrder->customer],
                    notification: new SalesOrderFulfilledNotification($this->fulfillment->salesOrder)
                );
            }
        }

        // Update the fulfillment status on sales order
        $this->fulfillment->salesOrder->updateFulfillmentStatus();

    }

    /**
     * @throws SalesOrderFulfillmentDispatchException
     * @throws Throwable
     * @throws BindingResolutionException
     */
    private function submitFulfillmentToShippingProvider(): void
    {
        try{
            ShipmentDispatcherFactory::make($this->fulfillment->fulfillment_type)
                ->dispatchFulfillmentToProvider($this->fulfillment);
        }catch (Throwable $e){
            // Delete the fulfillment so it can be retried later
            $this->fulfillment->delete();
            throw $e;
        }
    }
}
