<?php

namespace App\Jobs\Starshipit;

use App\Integrations\Starshipit;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Starshipit\StarshipitOrder;
use App\SDKs\ShipStation\ShipStationException;
use App\SDKs\Starshipit\StarshipitException;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;

class SubmitOrders implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /** @var array */
    protected $summary = ['success' => 0, 'fails' => 0, 'errors' => []];

    /**
     * Create a new job instance.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * Execute the job.
     *
     * @throws ShipStationException
     */
    public function handle(): void
    {
        // submit fulfillments to ShipStation
        SalesOrderFulfillment::with([])
            ->whereHas('salesOrder', function ($builder) {
                $builder->where('order_status', SalesOrder::STATUS_OPEN);
            })
            ->where('fulfillment_type', SalesOrderFulfillment::TYPE_STARSHIPIT)
            ->each(function (SalesOrderFulfillment $salesOrderFulfillment) {
                $submitStatus = static::submitOrderFulfillmentToStarshipit($salesOrderFulfillment, $this->integrationInstance);
                if ($submitStatus['success']) {
                    $this->summary['success']++;
                } else {
                    $this->summary['fails']++;
                    $this->summary['errors'][] = [
                        'salesOrderFulfillmentId' => $salesOrderFulfillment->id,
                        'message' => $submitStatus['error'],
                    ];
                }
            });

        // for logging
        $this->options = ['Integration Instance' => $this->integrationInstance->name.' ('.$this->integrationInstance->id.') '];
    }

    /**
     * Submit sales order fulfillment to Starshipit.
     */
    public static function submitOrderFulfillmentToStarshipit(SalesOrderFulfillment $salesOrderFulfillment, ?IntegrationInstance $starshipitInstance = null): array
    {
        if (! $starshipitInstance) {
            $starshipitInstance = IntegrationInstance::with([])->starshipit()->firstOrFail();
        }
        $starshipit = new Starshipit($starshipitInstance);

        $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['sku_fulfillment_id' => $salesOrderFulfillment->id]);

        // if the order exists, don't submit it again
        if ($starshipitOrder->exists && $starshipitOrder->order_id) {
            return ['success' => false, 'error' => 'AlreadyExists'];
        }

        try {
            $submitOrderResponse = $starshipit->submitOrder($salesOrderFulfillment->toStarshipitOrder());
            // Too Many Requests Error
            if ($submitOrderResponse->statusCode == 429) {
                sleep(2);
                $submitOrderResponse = $starshipit->submitOrder($salesOrderFulfillment->toStarshipitOrder());
            }

            if ($submitOrderResponse->statusCode == 200 && ($submitOrderResponse->body['success'] ?? true)) {
                $starshipitOrder->json_object = $submitOrderResponse->body['order'];
                // Don't store generated order id in the database
                //$starshipitOrder->order_id = $starshipitOrder->json_object['order_id'];
                $starshipitOrder->save();

                return ['success' => true];
            } else {
                if (($submitOrderResponse->body['errors'][0]['message'] ?? null) == 'Order Exists' || ($submitOrderResponse->body['errors'][0]['details'] ?? null) == 'Duplicate order number exists') {
                    return static::linkFulfillmentWithSSIOrder($salesOrderFulfillment, $starshipitInstance);
                }

                return ['success' => false, 'error' => $submitOrderResponse->body['errors'] ?? $submitOrderResponse->body];
            }
        } catch (Exception $exception) {
            return ['success' => false, 'error' => $exception->getMessage()];
        }
    }

    /**
     * Line the sales order fulfillment with a Starshipit order that has the same order_number
     */
    public static function linkFulfillmentWithSSIOrder(SalesOrderFulfillment $fulfillment, ?IntegrationInstance $starshipitInstance = null): array
    {
        if (! $starshipitInstance) {
            $starshipitInstance = IntegrationInstance::with([])->starshipit()->firstOrFail();
        }
        $starshipit = new Starshipit($starshipitInstance);

        try {
            $getOrderResponse = $starshipit->getOrder($fulfillment->fulfillment_number, 'order_number');
            // Too Many Requests Error
            if ($getOrderResponse->statusCode == 429) {
                sleep(2);
                $getOrderResponse = $starshipit->getOrder($fulfillment->fulfillment_number, 'order_number');
            }

            if ($getOrderResponse->statusCode == 200 && ($getOrderResponse->body['success'] ?? true)) {
                $ssiOrder = $getOrderResponse->body['order'];
                // check if it's archived
                // unshipped and archived
                if ($ssiOrder['archived']) {
                    return ['success' => false, 'error' => 'archived', 'order' => $ssiOrder];
                }
                // if the order status is "Printed/Shipped", the Starshipit always return archived=false even though it was archived
                if ($ssiOrder['status'] != 'Unshipped') {
                    // so, search on printed/shipping orders to check if it's not archived
                    $searchResponse = $starshipit->searchOrders(['phrase' => $fulfillment->fulfillment_number])->current();
                    if (empty($searchResponse->body['orders'])) {
                        return ['success' => false, 'error' => 'archived', 'order' => $ssiOrder];
                    }
                }

                // compare with lines
                $fulfillmentToSSIOrder = $fulfillment->toStarshipitOrder();

                if (count(($ssiOrder['items'] ?? [])) == count($fulfillmentToSSIOrder->items)) {
                    // the Starshipit order exists with the same order number, the same lines, and it's not archived

                    $starshipitOrder = StarshipitOrder::updateOrCreate(
                        ['order_id' => $ssiOrder['order_id']],
                        ['sku_fulfillment_id' => $fulfillment->id, 'json_object' => $ssiOrder]
                    );

                    if (($ssiOrder['manifested'] ?? false) == true) {
                        // mark as fulfilled and add the tracking info
                        (new GetTrackingJob($starshipitInstance, $fulfillment->id, false))->handle();
                    }

                    return ['success' => true, 'order' => $starshipitOrder, 'linked' => true];
                } else {
                    // the Starshipit order exists with the same order number, and it's not archived, but with different lines
                    return ['success' => false, 'error' => 'unfulfillable', 'order' => $ssiOrder];
                }
            } else {
                return ['success' => false, 'error' => $getOrderResponse->body['errors'] ?? $getOrderResponse->body];
            }
        } catch (\Throwable $exception) {
            return ['success' => false, 'error' => $exception->getMessage()];
        }
    }

    /**
     * check if the fulfillment is on Starshipit
     *
     *
     * @return false|StarshipitOrder|array false: doesn't exist, Order: already linked with SSI order, array: exists on SSI
     *
     * @throws StarshipitException
     */
    public function orderExistsOnStarshipit(SalesOrderFulfillment $fulfillment, ?IntegrationInstance $starshipitInstance = null): StarshipitOrder|bool|array
    {
        /*
         * TODO: I think we can check if the order already exists in the starshipit_orders, when we can't delete the SSI order,
         *  just keep it in starshipit_orders (soft delete)
         */

        // the fulfillment already linked with SSI order
        if ($ssiOrder = StarshipitOrder::query()->firstWhere('sku_fulfillment_id', $fulfillment->id)) {
            return ['status' => 'linked', 'order' => $ssiOrder];
        }

        if (! $starshipitInstance) {
            $starshipitInstance = IntegrationInstance::with([])->starshipit()->firstOrFail();
        }
        $starshipit = new Starshipit($starshipitInstance);

        $response = ['status' => 'not_exists', 'order' => null];
        // search by sales order number
        foreach ($starshipit->searchOrders(['phrase' => $fulfillment->salesOrder->sales_order_number, 'limit' => 250]) as $starshipitResponse) {
            $orders = collect($starshipitResponse->body['orders'])->filter(fn ($order) => Str::startsWith($order['order_number'], $fulfillment->salesOrder->sales_order_number));
            foreach ($orders as $order) {
                // the order is already linked with a sku fulfillment
                $startShipItQuery = StarshipitOrder::query();

                if ($startShipItQuery->whereNotNull('sku_fulfillment_id')->firstWhere('order_id', $order['order_id'])) {
                    continue;
                }

                // compare SSI order lines with the fulfillment lines
                $ssiOrder = $starshipit->getOrder($order['order_id'])->body['order'];
                $fulfillmentToSSIOrder = $fulfillment->toStarshipitOrder();
                if (count($ssiOrder['items']) == count($fulfillmentToSSIOrder->items)) {
                    // SSI order exists with the same order number prefix, same lines, and it's not linked with a sku fulfillment
                    return ['status' => 'exists', 'order' => $ssiOrder];
                } else {
                    // SSI order exists with the same order number prefix, and it's not linked with a sku fulfillment
                    $response = ['status' => 'unfulfillable', 'order' => $ssiOrder];
                }
            }
        }

        return $response;
    }
}
