<?php

namespace App\Jobs\Starshipit;

use App\Integrations\Starshipit;
use App\Models\IntegrationInstance;
use App\Models\Starshipit\StarshipitOrder;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class GetShippedOrdersJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance, protected array $options)
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $starshipit = new Starshipit($this->integrationInstance);

        foreach ($starshipit->getShippedOrders($this->options) as $response) {
            if ($response->body['success']) {
                $shippedOrders = collect($response->body['orders'])->where('manifest_sent', true); // Manifested (not just printed)
                if ($shippedOrders->isEmpty()) {
                    return;
                }

                $orders = null;

                $orders = StarshipitOrder::query();

                $orders->with('salesOrderFulfillment.salesOrder')
                    ->whereIn('order_id', $shippedOrders->pluck('order_id')->toArray())
                    ->each(function (StarshipitOrder $order) use ($shippedOrders) {
                        if (! $order->salesOrderFulfillment) {
                            return; // continue
                        }

                        $shippedOrder = $shippedOrders->firstWhere('order_id', $order->order_id);

                        $trackingInfo = [
                            'status' => 'Manifested', // at least the status is "Manifested" because we are fetched the Shipped orders
                            'carrier_name' => $shippedOrder['carrier_name'],
                            'carrier_service' => $shippedOrder['carrier_service_code'], // "code" because the webhook and tracking API return the "code" only
                            'tracking_number' => @$shippedOrder['tracking_number'],
                            'shipment_date' => $shippedOrder['shipped_date'],
                            'tracking_status' => $shippedOrder['tracking_short_status'] ?? 'Delivered',
                            'received_by' => 'RequestTrackingUpdatesService',
                            'price_breakdown' => $shippedOrder['price_breakdown'] ?? null,
                        ];

                        // update sales order fulfillment
                        GetTrackingJob::addTrackingToSalesOrderFulfillment($order->salesOrderFulfillment, $trackingInfo);
                        // add tracking data to the Starshipit order
                        GetTrackingJob::updateStarshipitOrder($order, $trackingInfo, false);
                    });
            }
        }
    }
}
