<?php

namespace App\Jobs\Shopify;

use App\Data\IntegrationInstanceInventoryData;
use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\ProductListingInventoryLocation;
use App\Notifications\MonitoringMessage;
use App\Queries\UpdateSalesChannelInventoryCache;
use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use PHPShopify\Exception\ApiException;
use PHPShopify\Exception\CurlException;
use PHPShopify\Exception\ResourceRateLimitException;
use Throwable;

class ShopifySyncInventoryJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected Throwable $exception;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance, protected int $page = 1)
    {
        $this->onQueue('syncInventory');
    }

    /**
     * Execute the job.
     *
     * @return array|void
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        echo $this->page;
        $shopify = new Shopify($this->integrationInstance);

        // the maximum cost points at the same time is 1000 cost points
        // and every product needs 10 cost points, so the limit must be 100

        $limit = 100;

        $salesChannelId = $this->integrationInstance->salesChannel->id;

        $productListingInventoryLocationsQuery = ProductListingInventoryLocation::with(['productListing.shopifyListing'])
            ->whereHas('productListing', function ($query) use ($salesChannelId) {
                $query->where('sales_channel_id', $salesChannelId)
                    ->whereNotNull('document_id');
            })
            ->whereNotNull('quantity')
            ->where(function (Builder $builder) {
                $builder->whereColumn('quantity', '!=', 'sales_channel_qty');
                $builder->orWhereNull('sales_channel_qty');
            })
            ->whereHas('productListing.shopifyListing', function ($query) {
                $query->where('removed_from_shopify', 0)->where('integration_instance_id', $this->integrationInstance->id);
            })
            // the page should be always 1 unless the first page is failed
            ->forPage($this->page ?: 1, $limit);

        $productListingInventoryLocationsQuery->get()->filter(function (ProductListingInventoryLocation $productListingInventoryLocation) {
            return abs($productListingInventoryLocation->sales_channel_qty ?? 0) > 100000;
        })->each(function (ProductListingInventoryLocation $productListingInventoryLocation) {
            $locationId = $productListingInventoryLocation->sales_channel_location_id;
            $productListing = $productListingInventoryLocation->productListing;
            dispatch(new ShopifyResetInventoryJob($this->integrationInstance, $productListing, $locationId));
        });

        // Only update the listings with sales channel qty < 100000.  The others get updated through ShopifyResetInventoryJob
        $productListingInventoryLocations = $productListingInventoryLocationsQuery->get()->filter(function (ProductListingInventoryLocation $productListingInventoryLocation) {
            return abs($productListingInventoryLocation->sales_channel_qty ?? 0) <= 100000;
        });

        if ($productListingInventoryLocations->isEmpty()) {
            return;
        }
        $locationIdMap = $productListingInventoryLocations->groupBy('sales_channel_location_id');
        customlog('syncShopifyInventory', 'Syncing inventory for '.$this->integrationInstance->salesChannel->name);
        foreach ($locationIdMap as $locationId => $productListingInventoryLocations) {
            $inventoryAdjustments = $productListingInventoryLocations->map(function (ProductListingInventoryLocation $productListingInventoryLocation) {
                /** @var ProductListingInventoryLocation $productListingInventoryLocation */
                $inventoryItemId = $productListingInventoryLocation->productListing->shopifyListing->json_object['inventory_item_id'];

                $delta = $productListingInventoryLocation->quantity - ($productListingInventoryLocation->sales_channel_qty ?? 0);

                if ($delta > 999999999) {
                    $delta = 999999999;
                }
                if ($delta < -999999999) {
                    $delta = -999999999;
                }

                customlog('syncShopifyInventory', 'Updating '.$productListingInventoryLocation->productListing->listing_sku.' with a change of '.$delta.' which should bring it from '.$productListingInventoryLocation->sales_channel_qty.' to '.$productListingInventoryLocation->quantity);

                return [

                    'availableDelta' => $delta,
                    'inventoryItemId' => "gid://shopify/InventoryItem/$inventoryItemId",
                ];
            })->unique('inventoryItemId')->values()->toArray();

            $variantIds = $productListingInventoryLocations->map(function (ProductListingInventoryLocation $productListingInventoryLocation) {
                return $productListingInventoryLocation->productListing->shopifyListing->variant_id;
            })->values()->toArray();

            $productListingInventoryLocationIds = $productListingInventoryLocations->map(function (ProductListingInventoryLocation $productListingInventoryLocation) {
                return $productListingInventoryLocation->id;
            })->values()->toArray();

            // send request on production mode
            if (config('app.env') !== 'production') {
                //return $inventoryAdjustments;
                return;
            }

            try {
                $response = $shopify->bulkAdjustQuantity($inventoryAdjustments, "gid://shopify/Location/$locationId");

                // some products can't update them, so we exclude them and send again
                if (! empty($errors = $response['data']['inventoryBulkAdjustQuantityAtLocation']['userErrors'])) {
                    // exclude failed indexes
                    foreach ($errors as $error) {
                        if (isset($error['field'][1])) {
                            $adjustmentNumber = $error['field'][1];

                            /*
                    * This means there was an error with the sync, most likely due to the inventory quantity being
                    * too great.  We can solve this by fetching the inventory value and updating the database so
                    * that the new delta next time won't go over the limit.  Then we update the adjustment.
                    */
                            try {
                                /** @var ProductListingInventoryLocation $productListingInventoryLocation */
                                $productListingInventoryLocation = ProductListingInventoryLocation::query()->find($productListingInventoryLocationIds[$adjustmentNumber]);

                                $productListingInventoryLocation->sales_channel_qty = $shopify->getInventoryLevel($productListingInventoryLocation->productListing->shopifyListing->json_object['inventory_item_id'], $locationId);
                                $productListingInventoryLocation->save();
                                $delta = $productListingInventoryLocation->quantity - $productListingInventoryLocation->sales_channel_qty;
                            } catch (Exception $e) {
                                Log::error($e->getMessage());

                                continue;
                            }

                            if ($delta > 999999999) {
                                $delta = 999999999;
                            }
                            if ($delta < -999999999) {
                                $delta = -999999999;
                            }
                            $inventoryAdjustments[$adjustmentNumber]['availableDelta'] = $delta;
                        }
                    }
                    // trying to send the request again without corrected indexes
                    $response = $shopify->bulkAdjustQuantity(array_values($inventoryAdjustments), "gid://shopify/Location/$locationId");
                    // if the error still exists, go to the next page
                    if (! empty($errors = $response['data']['inventoryBulkAdjustQuantityAtLocation']['userErrors'])) {
                        try {
                            $message = config('app.url')."\nShopify Sync Inventory Error\n".json_encode($errors)."\nvariantIds:".implode(',', $variantIds);
                            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));
                        } catch (Throwable) {
                        }

                        continue;
                    }
                }
            } catch (CurlException|ResourceRateLimitException|ApiException $shopifyException) {
                // here we still can't get the response to be able to calculate the delay time,
                // so we will delay to 1 minutes (exception in Shopify)
                try {
                    $message = config('app.url')."\nShopify Sync Inventory Exception\n".$shopifyException->getMessage();
                    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));
                } catch (Throwable) {
                }

                continue;
            } catch (Exception $e) {
                // Log the error message and the response if it exists
                $data = [
                    'message' => config('app.url')."\nShopify Sync Inventory Error occurred \n".$e->getMessage(),
                    'response' => $response ?? [],
                ];
                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage(json_encode($data)));

                continue;
            }

            $data = [];
            foreach ($response['data']['inventoryBulkAdjustQuantityAtLocation']['inventoryLevels'] as $inventoryLevel) {
                $inventoryItemId = (int) str_replace('gid://shopify/InventoryItem/', '', $inventoryLevel['item']['id']);
                $productVariantGid = 'gid://shopify/ProductVariant/';
                $data[$inventoryItemId] = '('.trim($inventoryLevel['item']['variant']['id'], $productVariantGid).','.$inventoryLevel['available'].',"'.Carbon::parse($inventoryLevel['updatedAt'])->timezone('UTC').'")';
            }

            try {
                $UpdateSalesChannelInventoryCache = new UpdateSalesChannelInventoryCache();
                $UpdateSalesChannelInventoryCache->execute($data, $this->integrationInstance->salesChannel->id, $locationId);
            } catch (Throwable $exception) {
                Log::debug('update ProductListing quantity after syncing:'.$exception->getMessage(), $exception->getTrace());
                $this->exception = $exception;
            }
        }
        /*
          * To be refactored later with multi location support
          */

        // we have more pages, so dispatch it again with the calculated delay time
        if ($productListingInventoryLocations->count() == $limit && isset($response['extensions'])) {
            dispatch(new self($this->integrationInstance, ++$this->page))
                ->delay(Shopify::getGraphqlTimeToRetry($response['extensions']['cost']))
                ->onQueue('syncInventory');
        }

        // last page
        if ($productListingInventoryLocations->count() < $limit) {
            $integrationSettings = $this->integrationInstance->integration_settings;
            $inventoryData = IntegrationInstanceInventoryData::from($integrationSettings['inventory'] ?? []);
            // set inventory last synced
            foreach ($inventoryData->locations as $key => $_) {
                $integrationSettings['inventory']['locations'][$key]['lastSyncedAt'] = now()->toISOString();
            }

            $this->integrationInstance->integration_settings = $integrationSettings;
            $this->integrationInstance->save();
        }

        // just to mark job as failed and to logging it
        if (! empty($this->exception)) {
            throw $this->exception;
        }
    }
}
