<?php

namespace App\Jobs\Shopify;

use App\Helpers;
use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

/**
 * Class ProcessOrder.
 */
class ShopifyProcessOrdersDb implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var array
     */
    protected $orders;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * @var Shopify
     */
    protected $shopify;

    /**
     * Maximum times to try canceled order handling.
     *
     * @var int
     */
    protected $maxCancelTries = 5;

    /**
     * @var array
     */
    protected $options = [];

    /**
     * ProcessOrders constructor.
     *
     * @param  IntegrationInstance|int  $integrationInstance
     */
    public function __construct(array $orders, $integrationInstance, array $options = [])
    {
        $this->orders = $orders;
        $this->integrationInstance = $integrationInstance instanceof IntegrationInstance ? $integrationInstance : IntegrationInstance::query()->findOrFail($integrationInstance);
        $this->options = $options;
    }

    public function handle(): void
    {
        set_time_limit(0);

        $this->shopify = new Shopify($this->integrationInstance);
        try {
            $this->handleOrders();
        } catch (Exception $e) {
            Log::debug($e->getMessage());
        }
    }

    protected function handleOrders()
    {
        foreach ($this->orders as $order) {
            $order_status = ShopifyOrder::getOrderStatus($order);

            // Cast necessary fields
            $order['total_price'] = (float) ($order['total_price']);
            // try {
            //     $order['transactions'] = $this->shopify->getOrderTransactions($order['id']);
            // } catch (\Exception $e) {
            //     Log::debug($e->getMessage());
            // }

            $shopifyOrder = ShopifyOrder::with([])->firstOrNew([
                'order_number' => $order['order_number'],
                'integration_instance_id' => $this->integrationInstance->id,
            ]);

            if ($shopifyOrder->exists) {
                $shopifyOrder->updated_by = $this->options['downloaded_by'] ?? ShopifyOrder::DOWNLOADED_BY_GET_ORDERS_JOB;
            } else {
                $shopifyOrder->downloaded_by = $this->options['downloaded_by'] ?? ShopifyOrder::DOWNLOADED_BY_GET_ORDERS_JOB;
            }

            $shopifyOrder->fill([
                'order_number' => $order['order_number'],
                'integration_instance_id' => $this->integrationInstance->id,
                'json_object' => $order,
            ])->save();

            /*
             * If open orders aren't set to be processed or the order date is before the open start date, skip processing
             */
            if ($order_status == 'open') {
                if (! $this->integrationInstance->open_start_date || ($this->integrationInstance->open_start_date && $this->integrationInstance->open_start_date > Carbon::parse($order['created_at'], 'UTC'))) {
                    $shopifyOrder->handleRefunds();

                    continue;
                }
            }

            /*
             * If closed orders aren't set to be processed or the order date is before the closed start date, skip processing
             */

            if ($order_status == 'closed' || $order_status == 'refunded') {
                if (! $this->integrationInstance->closed_start_date || ($this->integrationInstance->closed_start_date && $this->integrationInstance->closed_start_date > Carbon::parse($order['created_at'], 'UTC'))) {
                    $shopifyOrder->handleRefunds();

                    continue;
                }
            }

            // Create SKU order only if the shopify order was just created.
            if ($shopifyOrder->wasRecentlyCreated || ! $shopifyOrder->salesOrder) {
                $shopifyOrder->createSKUOrder();
            } elseif (! empty($changes = $shopifyOrder->getChanges())) {
                $shopifyOrder->updateSKUOrder($changes);
            }
        }

        // Cache orders
        if (! Helpers::isJobRunning(ShopifyCacheListingHasOrders::class)) {
            dispatch(new ShopifyCacheListingHasOrders())->onQueue('sales-channels');
        }
    }
}
