<?php

namespace App\Jobs\Shopify;

use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Queries\UpdateShopifyUnitCost;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use PHPShopify\Exception\CurlException;
use PHPShopify\Exception\ResourceRateLimitException;

/**
 * Class GetUnitCosts.
 */
class ShopifyGetUnitCosts implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * @var array
     */
    protected $options;

    /**
     * GetOrders constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance, array $options = [])
    {
        $this->integrationInstance = $integrationInstance;
        $this->options = array_merge(['limit' => 250], $options);
    }

    public function handle(): void
    {
        set_time_limit(0);

        $shopify = new Shopify($this->integrationInstance);

        $data = [];
        $removals = [];

        try {
            $ids = isset($this->options['ids']) ? explode(',', $this->options['ids']) :
                \App\Models\Shopify\ShopifyProduct::query()
                    ->whereNull('product')
                    ->whereNull('unit_cost')
                    ->where('integration_instance_id', $this->integrationInstance->id)
                    ->where('removed_from_shopify', 0)
                    ->pluck('inventory_item_id')
                    ->toArray();

            echo count($ids).' ids to process'."\n";
            foreach (array_chunk($ids, 250) as $chunkIds) {
                echo 'processing chunk ('.count($data).'/'.count($ids).')'."\n";
                $this->options['ids'] = implode(',', $chunkIds);

                $generator = $shopify->getInventoryItems($this->options);

                foreach ($generator as $response) {
                    $removals = array_merge($removals, array_diff($chunkIds, array_column($response, 'id')));

                    if (empty($response)) {
                        break;
                    }

                    foreach ($response as $inventoryItem) {
                        $inventoryItem['cost'] = $inventoryItem['cost'] ? $inventoryItem['cost'] : 0;
                        $data[] = '('.$inventoryItem['id'].','.$inventoryItem['cost'].')';
                    }
                }
            }

            $this->updateShopifyProducts($data, $removals);
        } catch (CurlException|ResourceRateLimitException $shopifyException) {
            // update the available data
            $this->updateShopifyProducts($data, $removals);
            // dispatch the job again to continue the remaining products
            $job = new static($this->integrationInstance, $this->options);
            $job->delay(now()->addMinutes(2)); // delay 2 minutes
            dispatch($job)
                ->onQueue($this->queue)
                ->onConnection($this->connection);
        }
        // dispatch even if the shopify exception thrown, because the CreateSkuProducts job will work on products that have unit_cost
        if (($this->integrationInstance->integration_settings['auto_create_products'] ?? false) === true) {
            dispatch(new \App\Jobs\Shopify\ShopifyCreateSkuProducts($this->integrationInstance))->onQueue($this->queue);
        }
    }

    /**
     * Update UnitCost and Removal
     */
    private function updateShopifyProducts(array $unitCostData, array $removals): void
    {
        $removals = array_unique($removals);
        $data_removals = [];

        echo count($removals).' products to mark as removed from shopify'."\n";
        foreach ($removals as $removal) {
            $data_removals[] = '('.$removal.')';
        }

        $UpdateShopifyUnitCost = new UpdateShopifyUnitCost();
        $UpdateShopifyUnitCost->execute($unitCostData, $this->integrationInstance->id);
    }
}
