<?php

namespace App\Jobs\Shopify;

use App\Enums\DownloadedBy;
use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Queries\UpdateShopifyListings;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;

class ShopifyGetProducts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    protected $summary = [
        'New Products' => 0,
        'Total Products' => 0,
    ];

    private static bool $hasInventoryLevelUpdateWebhook = false;

    /**
     * @var array
     */
    protected $options;

    /**
     * GetProducts constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance, array $options = [], bool $filterChangesByUpdatedAt = true)
    {
        $this->integrationInstance = $integrationInstance;
        $this->options = $options;
        if ($filterChangesByUpdatedAt) {
            Shopify::filterChangesByUpdatedAt();
        } else {
            Shopify::filterChangesByCreatedAt();
        }
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        self::$hasInventoryLevelUpdateWebhook = ShopifyWebhook::hasInventoryLevelUpdateWebhook($this->integrationInstance->id);

        $shopify = new Shopify($this->integrationInstance);

        $data_product_variants = [];
        $data_listings_update = [];

        foreach ($shopify->getProducts($this->options) as $products) {
            //print count($products) . ' products to update' . "\n";

            $variants = collect();
            foreach ($products as $product) {
                /*
                 * Remove variants that need removing
                 * This logic is good but needs to be in bulk query to perform all at once (not iterate through products)
                 */
                $data_product_variants[$product['id']] = array_column($product['variants'], 'id');

                $productVariants = static::extractProductVariations($this->integrationInstance, $product, $this->options['downloaded_by'] ?? DownloadedBy::Job);

                $variants = $variants->merge($productVariants);

                $this->summary['Total Products'] += 1;
            }

            $productVariants = null;

            // add/update to shopify_products table
            // chunks to avoid sql limit exception
            $variants->chunk(500)->each(function ($chunk) {
                ShopifyProduct::dataFeedBulkImport([
                    'data_to_import' => [
                        'type' => 'json',
                        'data' => $chunk->toJson(),
                    ],
                    'insert' => true,
                    'update' => true,
                    'sequence_id' => '_id',
                    'purgeDatabaseConnection' => false,
                    'mappings' => [
                        [
                            'expected_column_name' => 'variant_id',
                            'data_column_name' => 'variant_id',
                            'expected_column_type' => 'string',
                        ],
                        [
                            'expected_column_name' => 'removed_from_shopify',
                            'data_column_name' => 'removed_from_shopify',
                            'expected_column_type' => 'integer',
                        ],
                        [
                            'expected_column_name' => 'json_object',
                            'data_column_name' => 'json_object',
                            'expected_column_type' => 'json',
                        ],
                        [
                            'expected_column_name' => 'updated_by',
                            'data_column_name' => 'updated_by',
                            'expected_column_type' => 'string',
                        ],
                        [
                            'expected_column_name' => 'downloaded_by',
                            'data_column_name' => 'downloaded_by',
                            'expected_column_type' => 'string',
                        ],
                        [
                            'expected_column_name' => 'integration_instance_id',
                            'data_column_name' => 'integration_instance_id',
                            'expected_column_type' => 'integer',
                        ],
                        [
                            'expected_column_name' => 'updated_at',
                            'data_column_name' => 'updated_at',
                            'expected_column_type' => 'datetime',
                        ],
                        [
                            'expected_column_name' => 'created_at',
                            'data_column_name' => 'created_at',
                            'expected_column_type' => 'datetime',
                        ],
                    ],
                    'default_columns' => [

                    ],
                    'unique_by_columns' => [
                        'variant_id',
                        'integration_instance_id',
                    ],
                ]);

                ShopifyProduct::query()
                    ->where('integration_instance_id', $this->integrationInstance->id)
                    ->whereIn('variant_id', $chunk->pluck('variant_id'))
                    ->update(['_id' => DB::raw('`id`')]);

                // ShopifyProduct::query()->upsert($chunk->toArray(), [], ['json_object', 'updated_at', 'created_at', 'updated_by']);
            });

            foreach ($variants as $variant) {
                $json_object = json_decode($variant['json_object']);
                $data_listings_update[] = '('.$variant['variant_id'].",'".$json_object->sku."',".$json_object->inventory_quantity.','.(self::$hasInventoryLevelUpdateWebhook ? "'".now()->format('Y-m-d H:i:s')."'" : 'null').')';
            }

            // TODO: I think we should get rid from "_id", we should refactor the code and make a new mutator for it
            // set _id column
            // ShopifyProduct::query()
            //             ->where('integration_instance_id', $this->integrationInstance->id)
            //             ->whereIn('variant_id', $variants->pluck('variant_id'))
            //             ->update(['_id' => DB::raw('`id`')]);

            $variants = null;

            //print "Memory used: " . round(memory_get_usage(1)/1048576, 1) . "MB\n";
        }

        $data_product_variants_insert = [];

        foreach ($data_product_variants as $product_id => $variants) {
            foreach ($variants as $variant_id) {
                $data_product_variants_insert[] = '('.$variant_id.','.$product_id.')';
            }
        }

        $data_product_variants_insert = null;

        //print $num_removed . ' variations marked as removed' . "\n";

        //print "Memory used: " . round(memory_get_usage(1)/1048576, 1) . "MB\n";

        //print "Running shopify listing updater" . "\n";
        $UpdateShopifyListings = new UpdateShopifyListings();
        $UpdateShopifyListings->execute($data_listings_update, $this->integrationInstance->salesChannel->id);

        $data_listings_update = null;

        //print "Memory used: " . round(memory_get_usage(1)/1048576, 1) . "MB\n";

        dispatch(new ShopifyGetUnitCosts($this->integrationInstance))->onQueue($this->queue);
    }

    /**
     * Creates record by variation.
     */
    public static function extractProductVariations(IntegrationInstance $integrationInstance, array $payload, string $downloadedBy = DownloadedBy::Job): array
    {
        /* TODO: we should use Mysql model, also I think this query is wrong,
            because it links the product_id to line_items.id,
            but it should link variant_id to line_items.variant_id
            also, I believe we don't use has_orders attribute!
        */
        //        $payload['has_orders'] = Order::with([])
        //            ->where('integration_instance_id', $integrationInstance->id)
        //            ->where('line_items.id', $payload['id'])
        //            ->exists();

        $productVariants = [];
        $variants = $payload['variants'];
        unset($payload['variants']);

        foreach ($variants as $variant) {
            // Variants that don't require shipping don't need mapping and serve no purpose
            if (! $variant['requires_shipping']) {
                continue;
            }
            $variant['id'] = (int) $variant['id'];
            $variant['integration_instance_id'] = $integrationInstance->id;
            $variant['price'] = (float) $variant['price'];
            $variant['inventory_quantity'] = (int) $variant['inventory_quantity'];
            $variant['product_id'] = (int) $variant['product_id'];
            $variant['image'] = static::getVariantImage($variant, $payload);
            $variant['image_count'] = count($payload['images'] ?? []);
            $variant['variant_title'] = $variant['title'];
            $variant['removed_from_shopify'] = 0;
            if (isset($variant['unit_cost'])) {
                $variant['unit_cost'] = (float) $variant['unit_cost'];
            }
            if ($variant['variant_title'] == 'Default Title') {
                $variant['variant_title'] = $payload['title'];
            }
            unset($variant['title'], $variant['created_at'], $variant['updated_at']); // Ensure that product level updated_at is used.
            $variant['product_title'] = $payload['title'];

            $productVariants[] = [
                'variant_id' => $variant['id'],
                'integration_instance_id' => $integrationInstance->id,
                'removed_from_shopify' => $variant['removed_from_shopify'],
                'json_object' => json_encode(array_merge($payload, $variant)),
                'created_at' => static::dateToMysqlFormat($payload['created_at']),
                'updated_at' => static::dateToMysqlFormat($payload['updated_at']),
                'downloaded_by' => $downloadedBy,
                'updated_by' => $downloadedBy,
            ];
        }

        return $productVariants;
    }

    private static function dateToMysqlFormat(string $date): string
    {
        return Carbon::parse($date)->timezone(config('app.timezone'))->format('Y-m-d H:i:s');
    }

    /**
     * @return mixed|null
     */
    private static function getVariantImage(array $variant, array $payload)
    {
        if ($variant['image_id']) {
            $image = collect($payload['images'])->firstWhere('id', $variant['image_id']);
            if ($image) {
                return $image['src'];
            }
        }

        $image = $payload['image'];
        if (! $image && count($payload['images']) > 0) {
            $image = $payload['images'][0];
        }

        return $image ? $image['src'] : null;
    }

    public static function updateListingMapping(IntegrationInstance $integrationInstance, ShopifyProduct $product)
    {
        ProductListing::query()
            ->where('sales_channel_id', $integrationInstance->salesChannel->id)
            ->where('sales_channel_listing_id', $product['variant_id'])
            ->update([
                'document_id' => $product->id,
                'document_type' => ShopifyProduct::class,
                'listing_sku' => $product->sku,
                'sales_channel_qty' => $product->inventory_quantity,
                'sales_channel_qty_last_updated' => self::$hasInventoryLevelUpdateWebhook ? now() : null,
            ]);
    }

    private function addUnitCostToVariants(array &$products)
    {
        $shopify = new Shopify($this->integrationInstance);
        // get inventory items
        $ids = collect($products)->pluck('variants')->collapse()->pluck('inventory_item_id')->values();
        $inventoryItems = collect();
        // to fix "Too Large Request (HTTP Code 414)" error
        foreach ($ids->chunk(500) as $chuckIds) {
            foreach ($shopify->getInventoryItems(['ids' => $chuckIds->implode(','), 'limit' => 250]) as $items) {
                if (! isset($items['page_info'])) {
                    $inventoryItems = $inventoryItems->merge($items);
                }
            }
        }

        // add unit cost to the variants
        foreach ($products as &$product) {
            foreach ($product['variants'] as &$variant) {
                $inventoryItem = $inventoryItems->firstWhere('id', $variant['inventory_item_id']);
                $variant['unit_cost'] = $inventoryItem['cost'] ?? null;
            }
        }
    }
}
