<?php

namespace App\Jobs\Shopify;

use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\Middleware\WithoutOverlapping;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use PHPShopify\Exception\CurlException;
use PHPShopify\Exception\ResourceRateLimitException;

/**
 * Class GetOrders.
 */
class ShopifyGetOrdersJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected IntegrationInstance $integrationInstance;

    protected array $options;

    public int $tries = 5;

    public function middleware(): array
    {
        return [(new WithoutOverlapping('GetOrdersJob'))->releaseAfter(30)->expireAfter(240)];
    }

    /**
     * GetOrders constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance, array $options = [])
    {
        $this->integrationInstance = $integrationInstance;
        $this->options = array_merge(['limit' => 250], $options);
    }

    public function handle(): void
    {
        $shopify = new Shopify($this->integrationInstance);

        try {
            $response = $shopify->getSalesOrdersNew($this->options);
            $orders = @$response['orders'];

            if (!$orders) {
                return;
            }

            $nextPageParams = $response['nextPageParams'];
            //Log::info('Shopify orders downloading ('.count($orders).') for '.$this->integrationInstance->name, array_column($orders, 'name'));
            ShopifyDownloadOrder::bulkSaveShopifyOrderToSkuDB($this->integrationInstance->id, $orders, $this->options['downloaded_by'] ?? ShopifyOrder::DOWNLOADED_BY_GET_ORDERS_JOB);
        } catch (CurlException|ResourceRateLimitException $shopifyException) {
            //Log::info('Shopify GetOrders Exception: '.$shopifyException->getMessage());
            $this->release(120);
        }

        if ($nextPageParams) {
            //Log::info('250 Shopify orders downloaded for '.$this->integrationInstance->name.', moving on to next page');
            $this->options['page_info'] = $nextPageParams['page_info'];
            dispatch(new static($this->integrationInstance, $this->options))
                ->onQueue($this->queue)
                ->onConnection($this->connection);
        } else {
            //Log::info('Shopify order downloading complete for '.$this->integrationInstance->name);
            if (! isset($this->options['ids'])) {
                // We indicate the last time the orders were downloaded
                $this->integrationInstance->ordersLastDownloaded(now());
            }
        }
    }
}
