<?php

namespace App\Jobs\Shopify;

use App\Models\Shopify\ShopifyBulkOperation;
use App\Models\Shopify\ShopifyOrder;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ProcessShopifyOrdersJSONLJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(protected ShopifyBulkOperation $bulkOperation)
    {
        //
    }

    /*public function middleware(): array
    {
        return [(new WithoutOverlapping('ProcessShopifyOrdersJSONLJob'))->releaseAfter(60)->expireAfter(300)];
    }*/

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        $filepath = Storage::disk('shopify')->path('').$this->bulkOperation->reformatted_filename;

        /*
         * bulk query
         *
         * Create tmp table
         * LOAD DATA INTO tmp table
         * SELECT INTO OUTFILE records that are new
         *
         */

        $collate = config('database.connections.mysql.collation');

        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS temporary_shopify_orders (
                `shopify_order_id` bigint(20) AS (JSON_UNQUOTE(json_extract(json_object,'$.id'))) STORED,
                `json_object` json,
                UNIQUE INDEX (`shopify_order_id`)                                         
            ) ENGINE=INNODB DEFAULT COLLATE=$collate;
        SQL;

        DB::statement($createTemporaryTableQuery);

        $insertTemporaryTableQuery = "
            LOAD DATA LOCAL INFILE '".$filepath."'
                INTO TABLE temporary_shopify_orders
                FIELDS TERMINATED BY ','
                ENCLOSED BY '\"'
                ESCAPED BY '\\\\'
                LINES TERMINATED BY '\n'
                (`json_object`)
        ";

        DB::statement($insertTemporaryTableQuery);

        $updated_by = ShopifyOrder::DOWNLOADED_BY_GET_ORDERS_JOB;

        $updateShopifyOrdersQuery = <<<SQL
            UPDATE shopify_orders AS so
            INNER JOIN temporary_shopify_orders AS tso
            ON tso.shopify_order_id = so.shopify_order_id
            SET 
                so.json_object = tso.json_object,
                so.sku_updated_at = NOW(),
                so.updated_by = {$updated_by}
            WHERE so.integration_instance_id = {$this->bulkOperation->integration_instance_id}
        SQL;

        /*
         * TODO: See DownloadOrder.php
         *  there are fields that are not stored generated
         */

        DB::statement($updateShopifyOrdersQuery);

        $insertShopifyOrdersQuery = <<<SQL
            INSERT INTO sku.shopify_orders (
                                             `json_object`,
                                             `sku_created_at`,
                                             `sku_updated_at`
                                             )
            SELECT tso.`json_object`,
                   NOW(),
                   NOW()
            FROM temporary_shopify_orders AS tso
            LEFT JOIN shopify_orders AS so
            ON so.shopify_order_id = tso.shopify_order_id
            WHERE so.id IS NULL AND so.integration_instance_id = {$this->bulkOperation->integration_instance_id}
        SQL;

        DB::statement($insertShopifyOrdersQuery);
    }
}
