<?php

namespace App\Jobs\Shopify;

use App\Models\Shopify\ShopifyBulkOperation;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ProcessShopifyOrderTransactionsJSONLJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(protected ShopifyBulkOperation $bulkOperation)
    {
        //
    }

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        $filepath = Storage::disk('shopify')->path('').$this->bulkOperation->reformatted_filename;

        $collate = config('database.connections.mysql.collation');

        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS temporary_shopify_order_transactions (
                `shopify_order_id` bigint(20) AS (JSON_UNQUOTE(json_extract(json_object,'$.shopify_order_id'))) STORED,
                `transactions` json AS (JSON_UNQUOTE(json_extract(json_object,'$.transactions'))) STORED,
                `json_object` json,
                UNIQUE INDEX (`shopify_order_id`)                                         
            ) ENGINE=INNODB DEFAULT COLLATE=$collate;
        SQL;

        DB::statement($createTemporaryTableQuery);

        $insertTemporaryTableQuery = "
            LOAD DATA LOCAL INFILE '".$filepath."'
                INTO TABLE temporary_shopify_order_transactions
                FIELDS TERMINATED BY ','
                ENCLOSED BY '\"'
                ESCAPED BY '\\\\'
                LINES TERMINATED BY '\n'
                (`json_object`)
        ";

        DB::statement($insertTemporaryTableQuery);

        /*
         * TODO: Note we don't have a TransactionsHandler for UpdateOrder so updates to transactions will not have an impact
         *  on Shopify orders
         */
        $updateShopifyOrdersQuery = <<<SQL
            UPDATE shopify_orders AS so
            INNER JOIN temporary_shopify_order_transactions AS tsot
            ON tsot.shopify_order_id = so.shopify_order_id
            SET 
                so.transactions = tsot.transactions,
                so.sku_updated_at = NOW(),
                so.transactions_updated_at = NOW()
            WHERE so.integration_instance_id = {$this->bulkOperation->integration_instance_id}
        SQL;

        DB::statement($updateShopifyOrdersQuery);
    }
}
