<?php

namespace App\Jobs\Shopify;

use App\Enums\Shopify\ShopifyBulkOperationTypeEnum;
use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Repositories\Shopify\ShopifyTransactionRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

/**
 * Should be unique since only one allowed to run at a time on Shopify side
 */
class CreateShopifyGetTransactionsBulkOperationJob implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels, ShopifyBulkOperationTrait;

    public int $tries = 0;

    public int $uniqueFor = 60 * 5;

    public function uniqueId(): string
    {
        return 'shopify-bulk-transactions-'.$this->integrationInstance->id;
    }

    public function __construct(protected IntegrationInstance|int $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance instanceof IntegrationInstance ? $integrationInstance : IntegrationInstance::query()->findOrFail($integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        $shopify = new Shopify($this->integrationInstance);

        if (is_null($oldestDate = app()->make(ShopifyTransactionRepository::class)->getOldestUpdatedAtDateForTransactionsNeedingUpdate($this->integrationInstance)) |
            is_null($newestDate = app()->make(ShopifyTransactionRepository::class)->getNewestUpdatedAtDateForTransactionsNeedingUpdate($this->integrationInstance))) {
            return;
        }
        $oldestDate = Carbon::parse($oldestDate);
        $newestDate = Carbon::parse($newestDate);

        if (config('app.env') == 'production') {
            $response = $shopify->getOrdersTransactions(
                [],
                true,
                'updated_at:>='.
                "'".$oldestDate->toIso8601ZuluString()."'".
                ' AND '.
                'updated_at:<='.
                "'".$newestDate->toIso8601ZuluString()."'"
            );

            if (empty($response['data']['bulkOperationRunQuery'])) {
                Log::debug('Bulk operation error:', $response);

                return;
            }

            $this->handleBulkOperationResponse($response, ShopifyBulkOperationTypeEnum::OPERATION_TYPE_GET_ORDER_TRANSACTIONS);
        } else {
            $orderIds = app()->make(ShopifyTransactionRepository::class)->getOrderIdsForTransactionsNeedingUpdate($this->integrationInstance);
            $orderIds = array_slice($orderIds, 0, 50); // If not using bulk, we want to limit the # of transactions to 50 at a time
            $response = $shopify->getOrdersTransactions(
                $orderIds,
            );

            $orders = collect($response['data']['orders']['edges'])->pluck('node')->values();

            $order_transactions = [];
            $unified_transactions = [];
            foreach ($orders as $order) {
                $order_transactions[Shopify::getIdFromGID($order['id'])] = $order['transactions'];
            }

            foreach ($order_transactions as $order_id => $transactions) {
                foreach ($transactions as $transaction) {
                    $transaction['order_id'] = $order_id;
                    $unified_transactions[] = $transaction;
                }
            }

            app()->make(ShopifyTransactionRepository::class)->bulkSave($this->integrationInstance, $unified_transactions);
        }
    }
}
