<?php

namespace App\Jobs\Shopify;

use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use App\Services\StockTake\OpenStockTakeException;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use romanzipp\QueueMonitor\Traits\IsMonitored;

/**
 * Class CreateSalesOrderFromShopifyOrder.
 */
class CreateSalesOrderFromShopifyOrderJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, IsMonitored, Queueable, SerializesModels;

    /**
     * ProcessOrders constructor.
     */
    public function __construct(
        protected IntegrationInstance|int $integrationInstance,
        protected string $order_name,
        protected array $options = []
    ) {
        $this->onQueue('salesOrderProcessing');
        $this->integrationInstance = $integrationInstance instanceof IntegrationInstance ? $integrationInstance : IntegrationInstance::query()->findOrFail($integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        try {
            customlog('shopifyOrdersBenchmark', $this->order_name.': CreateSkuOrder starting');
            ShopifyOrder::getOrderFromOrderName($this->order_name, $this->integrationInstance)->createSKUOrder();
        } catch (OpenStockTakeException $openStockTakeException) {
            /** @see #SKU-5240 for this the job should just return, not fail.
             *      The idea is that stock takes are expected so shouldn’t cause any failures.
             *      Stock take locks shouldn’t last too long
             */
        }
    }
}
