<?php

namespace App\Jobs\ShipStation;

use App\Integrations\ShipStation;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\ShipStation\ShipstationOrder;
use App\SDKs\ShipStation\ShipStationException;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class SubmitOrders implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /** @var array */
    protected $summary = ['success' => 0, 'fails' => 0, 'errors' => []];

    /**
     * Create a new job instance.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * Execute the job.
     *
     * @throws ShipStationException
     */
    public function handle(): void
    {
        set_time_limit(0);
        // add fulfillments to sales order
        //    $this->autoFulfill();

        // submit fulfillments to ShipStation
        SalesOrderFulfillment::with([])
            ->whereHas('salesOrder', function ($builder) {
                $builder->where('order_status', SalesOrder::STATUS_OPEN);
            })
            ->where('fulfillment_type', SalesOrderFulfillment::TYPE_SHIPSTATION)
            ->chunk(100, function (Collection $salesOrderFulfillments) { // maximum of 100 per request
                $this->submitOrderFulfillmentsToShipStation($salesOrderFulfillments);
            });

        // for logging
        $this->options = ['Integration Instance' => $this->integrationInstance->name.' ('.$this->integrationInstance->id.') '];
    }

    /**
     * Bulk submit sales order fulfillmets to the ShipStation.
     *
     * @param  Collection|SalesOrderFulfillment[]  $salesOrderFulfillments
     *
     * @throws ShipStationException
     */
    private function submitOrderFulfillmentsToShipStation(Collection $salesOrderFulfillments)
    {
        $orderPrefix = config('shipstation.order_prefix');

        $shipStation = new ShipStation($this->integrationInstance);

        // sales order fulfillments that sent to ShipStation
        $shipStationOrdersIds = ShipstationOrder::with([])->whereIn('sku_fulfillment_id', $salesOrderFulfillments->pluck('id'))->get();

        $salesOrderFulfillments = $salesOrderFulfillments->whereNotIn('id', $shipStationOrdersIds->pluck('sku_fulfillment_id'));

        if ($salesOrderFulfillments->isEmpty()) {
            return;
        }
        $successIds = [];
        foreach ($salesOrderFulfillments->map->toShipStationOrder() as $order) {
            $sku_fulfillment_id = null;
            try {
                $shipstationOrder = $shipStation->submitOrder($order)->getResultAttribute()->toArray();

                $sku_fulfillment_id = $shipstationOrder['sku_fulfillment_id'] = intval(str_replace($orderPrefix, '', $shipstationOrder['orderKey']));
                $shipstationOrder['shipment'] = null;
                ShipstationOrder::with([])->create(['json_data' => $shipstationOrder, 'sku_fulfillment_id' => $sku_fulfillment_id]);
                $successIds[] = $shipstationOrder['sku_fulfillment_id'];
                $this->summary['success']++;
            } catch (\Exception $exception) {
                $this->summary['fails']++;
                $this->summary['errors'][] = [
                    'orderId' => $sku_fulfillment_id,
                    'message' => $exception->getMessage(),
                ];

                if ($exception->getCode() == 401) {
                    $this->integrationInstance->unauthorizedConnection();
                    break;
                }
            }
        }

        SalesOrder::with([])->whereHas('salesOrderFulfillments', function ($query) use ($successIds) {
            $query->whereIn('id', $successIds);
        })->update(['fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING]);
    }

    /**
     * Create new sales order fulfillments for sales orders that contains lines with auto-fulfillment warehouses.
     */
    private function autoFulfill()
    {
        $warehouses = $this->integrationInstance->integration_settings['fulfillment']['automatedWarehousesIds'] ?? [];

        SalesOrder::with([
            'salesOrderFulfillments',
            'salesOrderLines' => function ($query) use ($warehouses) {
                $query->whereIn('warehouse_id', $warehouses);
            },
            'salesOrderLines.salesOrderFulfillmentLines',
        ])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderLines', function ($query) use ($warehouses) {
                $query->whereIn('warehouse_id', $warehouses);
            })
            ->each(function (SalesOrder $salesOrder) {
                $groups = $salesOrder->salesOrderLines->groupBy('warehouse_id');

                foreach ($groups as $warehouseId => $lines) {
                    // check if the sales order fulfillment already created
                    if ($salesOrder->salesOrderFulfillments->firstWhere('warehouse_id', $warehouseId)) {
                        continue;
                    }

                    DB::transaction(function () use ($lines, $salesOrder, $warehouseId) {
                        // create sales order fulfillment
                        $salesOrderFulfillment = new SalesOrderFulfillment();
                        $salesOrderFulfillment->warehouse_id = $warehouseId;
                        $salesOrderFulfillment->requested_shipping_method_id = $salesOrder->shipping_method_id;
                        $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_SUBMITTED;
                        $salesOrderFulfillment->fulfillment_type = SalesOrderFulfillment::TYPE_SHIPSTATION;
                        $salesOrderFulfillment->fulfilled_at = now();
                        $salesOrder->salesOrderFulfillments()->save($salesOrderFulfillment);

                        // add sales order fulfillment lines
                        /** @var SalesOrderLine $line */
                        foreach ($lines as $line) {
                            $salesOrderFulfillment->salesOrderFulfillmentLines()->create([
                                'sales_order_line_id' => $line->id,
                                'quantity' => $line->quantity,
                            ]);
                        }

                        // mark sales order as fulfilled
                        FulfillSalesOrderService::make($salesOrder)->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at);
                    });
                }
            });
    }
}
