<?php

namespace App\Jobs\ShipStation;

use App\Integrations\ShipStation;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\ShipStation\ShipstationOrder;
use App\SDKs\ShipStation\Requests\ListOrdersRequest;
use App\SDKs\ShipStation\Responses\ListOrdersResponse;
use App\SDKs\ShipStation\ShipStationException;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class GetOrders implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    protected $listOrdersRequest;

    protected $summary = ['success' => 0, 'fails' => 0, 'errors' => []];

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->integrationInstance = IntegrationInstance::with([])->whereHas('integration', function ($query) {
            $query->where('name', Integration::NAME_SHIPSTATION);
        })->first();

        $this->listOrdersRequest = new ListOrdersRequest();
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $shipStation = new ShipStation($this->integrationInstance);

        /** @var ListOrdersResponse $orderResponse */
        try {
            foreach ($shipStation->getSalesOrders($this->listOrdersRequest) as $orderResponse) {
                foreach ($orderResponse->orders as $order) {
                    try {
                        $ord = ShipstationOrder::with([])->firstOrCreate(['orderId' => $order->orderId], ['json_data' => $order->toArray()]);

                        if ($ord->wasRecentlyCreated) {
                            $ord->shipment = null;
                            $ord->save();
                        }
                        $this->summary['success']++;
                    } catch (Exception $exception) {
                        $this->summary['fails']++;
                    }
                }
            }
        } catch (ShipStationException $shipStationException) {
            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                $this->integrationInstance->unauthorizedConnection();
            }
            throw $shipStationException;
        }
    }

    public function setRequestOptions(array $options)
    {
        $this->listOrdersRequest->setAttributes($options);
    }
}
