<?php

namespace App\Jobs\ShipStation;

use App\Http\Requests\FulfillSalesOrderRequest;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Notifications\MonitoringMessage;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Validation\ValidationException;

/**
 * @deprecated Use App\Services\SalesOrder\Fulfillments\AutomatedWarehouseFulfillment
 */
class AutoFulfillmentOrder implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /** @var SalesOrder */
    protected $salesOrder;

    /**
     * Create a new job instance.
     */
    public function __construct(SalesOrder $salesOrder)
    {
        $this->salesOrder = $salesOrder;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $automatedWarehouses = collect(Warehouse::getAutomatedWarehouses());

        /** @see #SKU-3592 entire order is fulfillable */
        if (! $this->salesOrder->is_fully_fulfillable) {
            return;
        }
        // lines that belongs to automated warehouses
        $groups = $this->salesOrder->salesOrderLines->whereIn('warehouse_id', $automatedWarehouses->pluck('warehouse_id'))->groupBy('warehouse_id');

        foreach ($groups as $warehouseId => $lines) {
            $integrationInstance = IntegrationInstance::with(['integration'])->find($automatedWarehouses->firstWhere('warehouse_id', $warehouseId)['integration_instance_id']);

            $fulfillRequestData = [
                'warehouse_id' => $warehouseId,
                'fulfilled_at' => now(),
                'fulfillment_type' => mb_strtolower($integrationInstance->integration->name), // shipstation or starshipit
                'requested_shipping_method_id' => $this->salesOrder->shipping_method_id,
                'requested_shipping_method' => $this->salesOrder->requested_shipping_method,
            ];

            // add lines that not fulfilled yet
            $fulfillRequestData['fulfillment_lines'] = $lines->map(function (SalesOrderLine $salesOrderLine) {
                return ['sales_order_line_id' => $salesOrderLine->id, 'quantity' => $salesOrderLine->unfulfilled_quantity];
            })->where('quantity', '>', 0)->values()->toArray();

            // all lines already fulfilled
            if (empty($fulfillRequestData['fulfillment_lines'])) {
                continue;
            }

            try {
                $fulfillRequest = FulfillSalesOrderRequest::createFromCustom($fulfillRequestData, 'POST', ['sales_order' => $this->salesOrder]);
                if (is_array($fulfillment = FulfillSalesOrderService::make($this->salesOrder)->fulfill($fulfillRequest))) {
                    Log::debug(__LINE__.": Failed fulfillment automated for {$this->salesOrder->sales_order_number}", $fulfillment);
                }
            } catch (\Throwable $exception) {
                Log::debug(__LINE__."Failed fulfillment automated for {$this->salesOrder->sales_order_number}: {$exception->getMessage()}", $exception instanceof ValidationException ? $exception->errors() : []);
            }
        }
    }

    /**
     * Auto fulfill the sales order if it's open and the warehouses of lines belong to the automated warehouses
     */
    public function fulfillOpenOrder(): void
    {
        try {
            if ($this->salesOrder->order_status == SalesOrder::STATUS_OPEN) {
                $this->handle();
            }
        } catch (\Throwable $exception) {
            try {
                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage("AutoFulfillmentOrder {$this->salesOrder->id}: {$exception->getMessage()}, {$exception->getFile()}:{$exception->getLine()}"));
            } catch (\Throwable $notificationException) {
                Log::debug(__LINE__."Failed fulfillment automated for {$this->salesOrder->sales_order_number}: {$exception->getMessage()}", $exception->getTrace());
            }
        }
    }
}
