<?php

namespace App\Jobs\Magento;

use App\Data\IntegrationInstanceInventoryData;
use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Queries\Magento\UpdateInventoryCache;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Optional;
use Throwable;

class SyncInventoryJob implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of seconds after which the job's unique lock will be released.
     */
    public int $uniqueFor = 60 * 60; // after one hour

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance, protected int $page = 1)
    {
    }

    /**
     * Execute the job.
     *
     * @return array|void
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        $sourceCode = $this->integrationInstance->integration_settings['inventorySources'][0]['magento_source_code'] ?? null;

        if (! $sourceCode) {
            throw new Exception('you should fetch magento inventory sources before.');
        }

        // cache product listings inventory for the mapped listings
        $this->cacheProductListingsInventory($sourceCode);

        // sync inventory
        ProductListing::query()
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->whereNotNull('quantity') // sku.io master stock
            ->whereNotNull('sales_channel_qty_last_updated')
            ->where(function (Builder $builder) {
                $builder->whereColumn('quantity', '!=', 'sales_channel_qty');
                $builder->orWhereNull('sales_channel_qty');
            })->chunk(20, function (Collection $chunk) use ($sourceCode) {
                /**
                 * Max is 20
                 *  {
                 *    "message": "Maximum items of type \"%type\" is %max",
                 *    "parameters": {"type":"\\Magento\\InventoryApi\\Api\\Data\\SourceItemInterface", "max":20}
                 *  }
                 */
                $sourceItems = collect();
                $chunk->each(function (ProductListing $productListing) use ($sourceCode, $sourceItems) {
                    $sourceItems->add([
                        'sku' => $productListing->listing_sku,
                        'quantity' => $productListing->quantity,
                        'source_code' => $sourceCode,
                        'status' => $productListing->quantity > 0 ? 1 : 0,
                    ]);
                });

                // send request on production mode
                if (config('app.env') !== 'production') {
                    // TODO: check the environment
                    //                              return false;
                }
                // sent source items to Magento
                (new Magento($this->integrationInstance))->updateSourceItems($sourceItems->toArray());
                // set sales_channel_qty from sku quantity
                ProductListing::query()->whereIn('id', $chunk->pluck('id'))->update(['sales_channel_qty' => DB::raw('`quantity`')]);
            });

        // set inventory last synced
        $integrationSettings = $this->integrationInstance->integration_settings;
        $inventoryData = IntegrationInstanceInventoryData::from($integrationSettings['inventory'] ?? []);
        if(!$inventoryData->locations instanceof Optional && $inventoryData->locations->count() > 0){
            $inventoryData->locations->first()->lastSyncedAt = now()->toISOString();
        }

        $integrationSettings['inventory'] = $inventoryData->toArray();
        $this->integrationInstance->integration_settings = $integrationSettings;
        $this->integrationInstance->save();
    }

    private function cacheProductListingsInventory(string $sourceCode)
    {
        $inventoryData = [];
        ProductListing::query()->where('sales_channel_id', $this->integrationInstance->salesChannel->id)->chunk(300, function (Collection $chunk) use ($sourceCode, &$inventoryData) {
            $magento = new Magento($this->integrationInstance);
            $options['pageSize'] = 300; // maximum
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'source_code',
                'conditionType' => 'eq',
                'value' => $sourceCode,
            ];
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'sku',
                'conditionType' => 'in',
                'value' => $chunk->implode('listing_sku', ','),
            ];
            foreach ($magento->getSourceItems($options) as $response) {
                foreach ($response['items'] as $sourceItem) {
                    /** @var ProductListing $productListing */
                    $productListing = $chunk->firstWhere('listing_sku', $sourceItem['sku']);
                    if (is_null($productListing->sales_channel_qty) || $productListing->sales_channel_qty != $sourceItem['quantity']) {
                        // TODO: how to get last inventory updated date
                        $inventoryData[] = "('{$productListing->id}', {$sourceItem['quantity']}, '".now()."')";
                    }
                }
            }
        });
        UpdateInventoryCache::execute($inventoryData, $this->integrationInstance->salesChannel->id);
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }
}
