<?php

namespace App\Jobs\Magento;

use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\ProductInventory;
use App\Models\ProductListing;
use App\SDKs\Magento\MagentoException;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class SyncExtendedInventoryInfoJob implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of seconds after which the job's unique lock will be released.
     */
    public int $uniqueFor = 60 * 60; // after one hour

    public $tries = 50;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $inventoryAttributeMappings = $this->integrationInstance->integration_settings['inventorySources'][0]['inventoryAttributeMappings'] ?? [];
        // check at least one listing has mapping
        $listingHasMappings = ProductListing::query()
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->where(function (Builder $builder) {
                $builder->whereRaw("json_extract(`inventory_rules`, '$.inventoryAttributeMappings.on_hand') != 'null'");
                $builder->orWhereRaw("json_extract(`inventory_rules`, '$.inventoryAttributeMappings.reserved') != 'null'");
                $builder->orWhereRaw("json_extract(`inventory_rules`, '$.inventoryAttributeMappings.incoming') != 'null'");
                $builder->orWhereRaw("json_extract(`inventory_rules`, '$.inventoryAttributeMappings.in_transit') != 'null'");
            })->exists();

        // not mapped
        if (empty(array_filter($inventoryAttributeMappings)) && ! $listingHasMappings) {
            return;
        }

        $updatedSkus = Cache::driver('file')->get($this->getUpdateSkusCacheKey(), []);
        $success = $this->productsNeedUpdateBuilder($inventoryAttributeMappings['incoming'] ?? null)->each(function ($item) use ($inventoryAttributeMappings, &$updatedSkus) {
            $listingInventoryAttributeMappings = $item['inventory_rules']['inventoryAttributeMappings'] ?? [];
            $attributeMappings = empty(array_filter($listingInventoryAttributeMappings)) ? $inventoryAttributeMappings : $listingInventoryAttributeMappings;

            $customAttributes = [];
            // only send mapped inventory
            foreach (array_filter($attributeMappings) as $inventory => $attributeCode) {
                $customAttributes[] = ['attribute_code' => $attributeCode, 'value' => $item[$inventory]];
            }

            try {
                (new Magento($this->integrationInstance))->updateProduct($item['sku'], ['custom_attributes' => $customAttributes]);
                $updatedSkus[] = $item['sku'];
            } catch (MagentoException $exception) {
                if (Str::startsWith($exception->getMessage(), 'cURL Error')) {
                    // save the updated skus to exclude in the next job
                    Cache::driver('file')->forever($this->getUpdateSkusCacheKey(), $updatedSkus);

                    return false;
                }
                throw $exception;
            }
        });

        // failed
        if ($success === false) {
            $this->release($this->backoff());

            return;
        }

        // forget the updated skus
        Cache::driver('file')->forget($this->getUpdateSkusCacheKey());
    }

    /**
     * Query builder to products that need update the extended inventory info
     */
    private function productsNeedUpdateBuilder(?string $incomingAttributeCode): Builder
    {
        $instanceInventoryRules = json_encode($this->integrationInstance->integration_settings['inventorySources'][0] ?? []);

        return ProductListing::query()
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->whereNotNull('quantity') // sku.io master stock
            ->whereNotNull('sales_channel_qty_last_updated')
            ->where(function (Builder $builder) use ($incomingAttributeCode) {
                // to check the on_hand, reserved, incoming and in_transit
                $builder->where(function (Builder $builder) {
                    $builder->whereColumn('quantity', '!=', 'sales_channel_qty');
                    $builder->orWhereNull('sales_channel_qty');
                });
                // to check th incoming
                if ($incomingAttributeCode) {
                    $builder->orWhereRaw("IFNULL(`inventory_incoming`, 0) != {$this->getMagentoCustomAttributeValueQuery($incomingAttributeCode)}");
                }
            })
            ->when($incomingAttributeCode, function (Builder $builder) {
                $builder->join('magento_products', 'magento_products.product', '=', 'product_listings.id');
            })
                     // exclude updated skus
            ->when($this->updatedSkusTable(), function (Builder $builder) {
                $builder->whereNotExists(function ($builder) {
                    $builder->select('*')->from('updated_skus')->whereColumn('updated_skus.sku', '=', 'product_listings.listing_sku');
                });
            })
            ->leftJoinSub(
                ProductInventory::query()
                    ->join('product_listings', function (JoinClause $join) use ($instanceInventoryRules) {
                        $join->on('product_listings.product_id', '=', 'products_inventory.product_id')
                            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
                            ->whereRaw("JSON_CONTAINS(JSON_EXTRACT(IFNULL(`inventory_rules`, '$instanceInventoryRules'), '$.selectedWarehouses'),CAST(products_inventory.warehouse_id AS JSON))");
                    })
                    ->groupBy('products_inventory.product_id')
                    ->select([
                        'products_inventory.product_id',
                        DB::raw('sum(`inventory_total`) inventory_total'),
                        DB::raw('sum(`inventory_reserved`) inventory_reserved'),
                        DB::raw('sum(`inventory_in_transit`) inventory_in_transit'),
                    ]), 'pi', 'pi.product_id', 'product_listings.product_id')
            ->select([
                'product_listings.listing_sku as sku',
                'product_listings.inventory_rules',
                DB::raw('IFNULL(inventory_total, 0) as on_hand'),
                DB::raw('IFNULL(inventory_reserved, 0) as reserved'),
                DB::raw('IFNULL(inventory_in_transit, 0) as in_transit'),
            ]);
    }

    /**
     * Sql query to get the value of the custom attribute from json_object of the magento product
     */
    private function getMagentoCustomAttributeValueQuery(string $attributeCode): string
    {
        return "IFNULL(json_unquote(json_extract(`json_object`, JSON_UNQUOTE(replace(json_search(json_object, 'one', '$attributeCode', NULL, '$.custom_attributes[*].attribute_code'), 'attribute_code', 'value')))), 0)";
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }

    /**
     * Calculate the number of seconds to wait before retrying the job.
     */
    public function backoff(): int
    {
        return $this->attempts() * 60;
    }

    private function getUpdateSkusCacheKey(): string
    {
        return "SyncExtendedInventoryInfoJob:{$this->integrationInstance->id}:skus";
    }

    /**
     * Create and Insert updated skus
     *
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    private function updatedSkusTable(): bool
    {
        $updatedSkus = Cache::driver('file')->get($this->getUpdateSkusCacheKey(), []);
        if (empty($updatedSkus)) {
            return false;
        }

        $updatedSkus = array_map(fn ($sku) => "('$sku')", array_unique($updatedSkus));

        $updatedSkusTable = 'CREATE TEMPORARY TABLE updated_skus (`sku` varchar(255), PRIMARY KEY (`sku`)) ENGINE=InnoDB;';

        $insertSkus = 'INSERT INTO updated_skus VALUES '.implode(',', $updatedSkus).';';

        DB::statement($updatedSkusTable);
        DB::insert($insertSkus);

        return true;
    }
}
