<?php

namespace App\Jobs\Magento;

use App\Helpers;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Notifications\MonitoringMessage;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class MapSalesOrderLinesJob implements ShouldBeUnique, ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of seconds after which the job's unique lock will be released.
     */
    public int $uniqueFor = 60 * 30;

    private string $temporaryOrderLinesTable;

    private string $temporaryOrderIdsTable;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            DB::beginTransaction();

            $this->insertSalesOrderLinesIntoTempTable();

            // use warehouse_priority setting to set line warehouses
            $priorityWarehouses = json_decode(Helpers::setting(Setting::KEY_WAREHOUSE_PRIORITY), true);
            $setFirstWarehouseQuery = '';
            // if the warehouse_priority does not set, use the first warehouse
            if (empty($priorityWarehouses)) {
                $firstWarehouseId = Warehouse::with([])->value('id');
                if (! $firstWarehouseId) {
                    throw new \Exception('The system does not have any warehouse');
                }
                // use first warehouse id to update sales order line warehouse_id
                $setFirstWarehouseQuery = ", sol.warehouse_id = $firstWarehouseId";
            }

            // map sales order lines with products
            $query = <<<SQL
                UPDATE sales_order_lines sol
                    INNER JOIN {$this->temporaryOrderLinesTable} tsol ON tsol.sales_order_line_id = sol.id
                SET sol.product_listing_id = tsol.product_listing_id,
                    sol.product_id = tsol.product_id
                    $setFirstWarehouseQuery
            SQL;
            DB::statement($query);

            // set the warehouse_id to the mapped lines
            if (! empty($priorityWarehouses)) {
                // actually, these two queries only for existing skus
                $this->setWarehouseToLinesByPriorityWarehouses($priorityWarehouses);
                $this->setWarehouseToLinesBySupplierInventory();
                // this query that will actually be used to new products, because the product does not have an inventory
                $this->setWarehouseToLinesByFirstPriorityWarehouse($priorityWarehouses);
            }

            // add sales orders that need to handle into a temporary table
            $this->insertSalesOrdersIntoTempTable();
            // handle sales orders
            Model::preventLazyLoading();
            $salesOrders = SalesOrder::with([
                'salesChannel.integrationInstance.integration',
                'magentoOrder.integrationInstance',
                'currency',
                'salesOrderLines.backorderQueue',
                'salesOrderLines.product.defaultSupplierProduct',
                'salesOrderLines.warehouse',
                'salesOrderLines.inventoryMovements',
                'salesOrderLines.salesOrderFulfillmentLines',
                'salesOrderLines.activeBackorderQueue',
            ])
                ->join($this->temporaryOrderIdsTable, "{$this->temporaryOrderIdsTable}.sales_order_id", '=', 'sales_orders.id');

            echo "Orders count: {$salesOrders->count()}\n";

            $salesOrders->each(function (SalesOrder $salesOrder) {
                echo 'Processing sales order '.$salesOrder->sales_order_number."\n";
                $salesOrder->salesOrderLines->map->setRelation('salesOrder', $salesOrder->withoutRelations());

                MapSalesOrderLines::handleAfterMapLines(
                    $salesOrder,
                    $salesOrder->magentoOrder,
                    null,
                    false
                );
            });
            Model::preventLazyLoading(false);

            DB::commit();
        } catch (\Throwable $exception) {
            DB::rollBack();
            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage("Can't handle sales orders: {$exception->getMessage()}"));
            Log::debug("Can't handle sales orders {$exception->getMessage()}", $exception->getTrace());
            throw $exception;
            // TODO: support partially map, exclude lines that thrown an exception(unmap it)
        }
    }

    private function insertSalesOrderLinesIntoTempTable()
    {
        // create a temporary table
        $collate = config('database.connections.mysql.collation');
        $this->temporaryOrderLinesTable = 'temporary_sales_order_lines_'.now()->timestamp;
        $createTempOrderLinesTableQuery = <<<SQL
            CREATE TEMPORARY TABLE {$this->temporaryOrderLinesTable} (
                `sales_order_line_id` bigint(20) unsigned,
                `sales_order_id` bigint(20) unsigned,
                `product_id` bigint(20) unsigned,
                `product_listing_id` bigint(20) unsigned,
                `warehouse_id` bigint(20) unsigned,
                `quantity` int,
                PRIMARY KEY (`sales_order_line_id`)
            ) ENGINE=InnoDB COLLATE={$collate}
        SQL;
        DB::statement($createTempOrderLinesTableQuery);

        // insert sales order lines into the temporary table
        $insertQuery = <<<SQL
            INSERT INTO $this->temporaryOrderLinesTable (sales_order_line_id, sales_order_id, product_id, product_listing_id, quantity)
            SELECT sol.id, sol.sales_order_id, pl.product_id, pl.id, sol.quantity FROM sales_order_lines sol 
                INNER JOIN magento_order_line_items moli ON moli.line_id = sol.sales_channel_line_id
                INNER JOIN magento_orders mo ON mo.id = moli.magento_order_id and mo.integration_instance_id = ? 
                INNER JOIN magento_products mp ON mp.variant_id = moli.product_id and mp.integration_instance_id = ?
                INNER JOIN product_listings pl ON mp.product = pl.id
            WHERE sol.product_id IS NULL AND mp.product IS NOT NULL AND sol.sales_channel_line_id REGEXP "^([0-9])+$"
        SQL;
        DB::insert($insertQuery, [$this->integrationInstance->id, $this->integrationInstance->id]);
    }

    private function insertSalesOrdersIntoTempTable()
    {
        // create the temporary table
        $collate = config('database.connections.mysql.collation');
        $this->temporaryOrderIdsTable = 'temporary_sales_order_ids_'.now()->timestamp;
        $createTempOrderIdsTableQuery = <<<SQL
            CREATE TEMPORARY TABLE {$this->temporaryOrderIdsTable} (
                `sales_order_id` bigint(20) unsigned, 
                PRIMARY KEY (`sales_order_id`)
            ) ENGINE=InnoDB COLLATE={$collate} AS (
                SELECT DISTINCT `sales_order_id` FROM {$this->temporaryOrderLinesTable}
            )
        SQL;
        DB::statement($createTempOrderIdsTableQuery);
    }

    private function setWarehouseToLinesByPriorityWarehouses(array $priorityWarehouses)
    {
        $priorityWarehouses = implode(',', $priorityWarehouses);
        $query = <<<SQL
            UPDATE sales_order_lines sol2 INNER JOIN (
                SELECT sol.id,
                       pi.warehouse_id,
                       ROW_NUMBER() OVER (PARTITION BY sol.id ORDER BY FIELD(pi.warehouse_id, $priorityWarehouses)) priority
                FROM sales_order_lines sol
                    INNER JOIN {$this->temporaryOrderLinesTable} tsol ON tsol.sales_order_line_id = sol.id
                    INNER JOIN products_inventory pi ON sol.product_id = pi.product_id AND pi.warehouse_id IN ($priorityWarehouses) AND pi.inventory_available >= sol.quantity
                WHERE sol.warehouse_id IS NULL) lpw ON lpw.id = sol2.id AND lpw.priority = 1
            SET sol2.warehouse_id = lpw.warehouse_id
        SQL;
        DB::update($query);
    }

    private function setWarehouseToLinesBySupplierInventory()
    {
        $query = <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryOrderLinesTable} tsol ON tsol.sales_order_line_id = sol.id
                INNER JOIN supplier_products sp ON sol.product_id = sp.product_id AND sp.is_default = 1
                INNER JOIN supplier_inventory si ON sol.product_id = si.product_id AND sp.supplier_id = si.supplier_id
                INNER JOIN warehouses w ON w.id = si.warehouse_id AND w.supplier_id IS NOT NULL AND w.dropship_enabled = 1
            SET sol.warehouse_id = si.warehouse_id
            WHERE sol.warehouse_id IS NULL AND (si.in_stock = 1 OR si.quantity >= sol.quantity)
        SQL;
        DB::update($query);
    }

    private function setWarehouseToLinesByFirstPriorityWarehouse(array $priorityWarehouses)
    {
        $firstPriorityWarehouse = $priorityWarehouses[0];

        $query = <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryOrderLinesTable} tsol ON tsol.sales_order_line_id = sol.id
            SET sol.warehouse_id = $firstPriorityWarehouse
            WHERE sol.warehouse_id IS NULL
        SQL;
        DB::update($query);
    }

    protected function getHighestPriorityWarehouseId(): int
    {
        $priorityWarehouses = json_decode(Setting::getValueByKey(Setting::KEY_WAREHOUSE_PRIORITY), true);

        return $priorityWarehouses[0];
    }

    /**
     * The unique ID of the job.
     */
    public function uniqueId(): string
    {
        return $this->integrationInstance->id;
    }

    /**
     * Get the cache driver for the unique job lock.
     */
    public function uniqueVia(): Repository
    {
        return Cache::driver('redis');
    }
}
