<?php

namespace App\Jobs\Magento;

use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\Magento\InventorySource;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class GetInventorySourcesJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected IntegrationInstance $integrationInstance)
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $shopify = new Magento($this->integrationInstance);
        $sourceCodes = [];
        foreach ($shopify->getInventorySources() as $response) {
            $sources = array_map(fn ($source) => ['integration_instance_id' => $this->integrationInstance->id, 'json_object' => json_encode($source)], $response['items']);
            InventorySource::query()->upsert($sources, [], ['json_object']);
            $sourceCodes = array_merge(array_column($response['items'], 'source_code'));
        }

        InventorySource::query()
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->whereNotIn('source_code', $sourceCodes)
            ->delete();

        // Add inventory sources to the integration instance
        $this->setInventorySourcesToIntegrationInstance();
    }

    /**
     * Add inventory sources to the integration instance.
     *
     * @see SKU-3453
     */
    private function setInventorySourcesToIntegrationInstance()
    {
        $integrationSettings = $this->integrationInstance->integration_settings;
        $existedInventoryLocations = collect($integrationSettings['inventory']['locations'] ?? []);
        // to delete the removed/inactive magento sources
        $inventoryLocations = collect([]);

        InventorySource::query()->where('integration_instance_id', $this->integrationInstance->id)->each(function (InventorySource $inventorySource) use ($inventoryLocations, $existedInventoryLocations) {
            if (! $inventorySource->json_object['enabled']) {
                return;
            }

            // If the downloaded source already exists, don’t change any master of stock/warehouse/etc. settings
            if ($existedInventoryLocation = $existedInventoryLocations->firstWhere('magento_source_code', $inventorySource->source_code)) {
                $inventoryLocations->add($existedInventoryLocation);
            } else {
                /**
                 * If the source does not already exist, initialize with:
                 *  - master of stock: none
                 *  - warehouses: none.
                 */
                $inventoryLocations->add([
                    'magento_source_code' => $inventorySource->source_code,
                    'magento_source_name' => $inventorySource->name,
                    'masterOfStock' => IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER,
                    'selectedWarehouses' => [],
                    'inventoryAttributeMappings' => [
                        'on_hand' => null,
                        'reserved' => null,
                        'incoming' => null,
                        'in_transit' => null,
                    ],
                ]);
            }
        });
        // save integration sources
        $integrationSettings['inventorySources'] = $inventoryLocations->values()->toArray();
        $this->integrationInstance->integration_settings = $integrationSettings;
        $this->integrationInstance->save();
    }
}
