<?php

namespace App\Jobs\Magento;

use App\Integrations\Magento;
use App\Models\Currency;
use App\Models\IntegrationInstance;
use App\Models\Magento\CustomerGroup;
use App\Models\ProductPricingTier;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;

class GetCustomerGroupsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(public IntegrationInstance $integrationInstance)
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $magento = new Magento($this->integrationInstance);

        foreach ($magento->getCustomerGroups() as $response) {
            array_map(fn ($customerGroup) => $this->handleCustomerGroup($customerGroup), $response['items']);
        }
    }

    private function handleCustomerGroup(array $customerGroup)
    {
        // exclude “NOT LOGGED IN”
        if ($customerGroup['code'] == 'NOT LOGGED IN') {
            return false;
        }

        /** @var CustomerGroup $cg */
        $cg = CustomerGroup::with(['productPricingTier'])->firstWhere(['integration_instance_id' => $this->integrationInstance->id, 'customer_group_id' => $customerGroup['id']]);

        if ($cg) {
            // create a new product pricing tier if it is not exist
            if (! $cg->productPricingTier) {
                $pricingTier = ProductPricingTier::query()->firstOrCreate(
                    ['name' => $customerGroup['code']],
                    ['name' => $customerGroup['code'], 'currency_code' => Currency::default()->code]
                );
                $cg->product_pricing_tier_id = $pricingTier->id;
            }
            // TODO: should we update the name if it is changed from Magento or from SKU?

            return $cg->update(Arr::except($customerGroup, 'id'));
        }

        // add a new product pricing tier
        $pricingTier = ProductPricingTier::query()->firstOrCreate(
            ['name' => $customerGroup['code']],
            ['name' => $customerGroup['code'], 'currency_code' => Currency::default()->code]
        );
        // add to magento customer group
        CustomerGroup::query()->updateOrCreate(
            ['integration_instance_id' => $this->integrationInstance->id, 'customer_group_id' => $customerGroup['id']],
            Arr::except($customerGroup, 'id') + ['product_pricing_tier_id' => $pricingTier->id]
        );
    }
}
