<?php

namespace App\Jobs\Magento;

use App\Importers\Parsers\FieldParserFactory;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\Attribute;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\Magento\Product;
use App\Models\Magento\ProductAttribute;
use App\Models\ProductPricingTier;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\Warehouse;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class CreateSkuProducts.
 */
class CreateSkuProducts implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected IntegrationInstance $integrationInstance;

    protected array $options;

    private Collection $mappings;

    private string $temporaryProductsTable = 'temporary_magento_products';

    private string $temporaryTagsTable = 'temporary_magento_tags';

    /**
     * GetOrders constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    public function handle(): void
    {
        set_time_limit(0);

        $this->mappings = collect(DataImportMapping::query()
            ->where('model', 'listings')
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->value('mapping'));
        if ($this->mappings->isEmpty()) {
            throw new \Exception('the integration instance does not have mapping rules.');
        }
        // sku field
        if (! $skuMapping = $this->mappings->firstWhere('sku_field', 'sku')) {
            throw new \Exception('the mapping does not have sku field');
        }
        $skuListingField = $skuMapping['listing_field'];
        if (! empty($parsers = $skuMapping['parsers'])) {
            foreach ($parsers as $parser) {
                $skuListingField = FieldParserFactory::make($parser['rule'], $parser['args'], $this)->parseSqlQuery($skuListingField, 'magento_products');
            }
        } else {
            $skuListingField = "`magento_products`.`$skuListingField`";
        }

        // get the mapped supplier pricing tier
        $supplierPricingTier = null;
        if ($supplierPricingMapping = $this->mappings->filter(fn ($m) => preg_match("/supplier_pricing\..*\.value/", $m['sku_field']))->first()) {
            $supplierPricingTierName = explode('.', $supplierPricingMapping['sku_field'])[1];
            $supplierPricingTier = SupplierPricingTier::query()->firstWhere('name', $supplierPricingTierName);
        }

        $count = 3000;
        do {
            $productsToCreate = Product::query()
                ->select('magento_products.*', 'products.id as sku_product_id')
                ->where('integration_instance_id', $this->integrationInstance->id)
                ->whereNull('product')
                ->whereRaw(DB::raw("$skuListingField != ''")->getValue(DB::getQueryGrammar()))
                ->whereIn('type_id', ['simple', 'bundle'])
                ->where('status', 1) // enabled
                // existing skus just link them to product_listings and map sales order lines
                ->leftJoin('products', 'products.sku', '=', DB::raw($skuListingField))
                ->orderBy('magento_products.type_id', 'desc') // create simple products first
                ->forPage(1, $count)->get();

            $magentoProductAttributes = ProductAttribute::query()->where('integration_instance_id', $this->integrationInstance->id)->get()->keyBy('code');
            $dataInsertProducts = [];
            // parse products data to temporary tables data
            $productsToCreate->each(function (Product $product) use ($magentoProductAttributes, &$dataInsertProducts) {
                $product->setRelation('integrationInstance', $this->integrationInstance);
                $product->setRelation('productAttributes', $magentoProductAttributes);
                //                print ($product->sku_product_id ? 'Linking ' : 'Creating ') . $product->sku . "\n";
                $insertProduct = [
                    'magento_product_id' => $product->id,
                    'magento_product_title' => $product->name,
                    'magento_product_price' => $product->price,
                    'magento_product_sku' => $product->sku,
                    'magento_type_id' => $product->type_id,
                    'variant_id' => $product->variant_id,
                    'product_id' => $product->sku_product_id,
                    'product_exists' => is_null($product->sku_product_id) ? 0 : 1,
                ];

                foreach ($this->mappings as $mapping) {
                    $listingFieldValue = $product->parseRules($product->getValue($mapping['listing_field']), $mapping['parsers'] ?? []);
                    $insertProduct[$this->temporaryColumnName($mapping['listing_field'])] = is_null($listingFieldValue) ? null : trim($listingFieldValue);
                }

                if ($product->type_id == 'bundle') {
                    $insertProduct['component_skus'] = collect($product->json_object['extension_attributes']['bundle_product_options'])->map(fn ($c) => $c['product_links'][0]['sku'] ?? null)->filter()->values()->toArray();
                }

                $dataInsertProducts[] = $insertProduct;
            });

            // exclude bundle products that have an unmapped component(disabled)
            $dataInsertProducts = $this->filterBundleProductsToCreate($dataInsertProducts);

            if (empty($dataInsertProducts)) {
                break;
            }

            DB::transaction(function () use ($supplierPricingMapping, $supplierPricingTier, $dataInsertProducts) {
                /**
                 * if we notice a performance issue,
                 * we can create other temporary tables for images, pricing, attributes and others
                 * currently, the query adds one attribute(for example) to all products
                 */

                // create the temporary table and insert products data
                $this->insertProductsDataToTemporaryTable($dataInsertProducts);

                // create product brands and add brand_ids into the temporary table
                $this->addProductBrands();

                // insert products with basic fields
                $this->insertProducts();

                // Product listings, to use product_listings when adding bundle components
                $this->insertProductListings();

                // add bundle components, here to exclude bundle products that have components still are not mapped(or disabled)
                $this->addBundleComponents();

                // add product images
                $this->addProductImages();

                // add default supplier to products
                $this->addDefaultSupplierToProducts($supplierPricingTier, $supplierPricingMapping['listing_field'] ?? null);

                // add pricing to products
                $this->addPricingToProducts();

                // add attributes to products
                $this->addAttributesToProducts();

                // add main category to products
                $this->addMainCategoryToProducts();

                // map sales order lines
                MapSalesOrderLinesJob::dispatch($this->integrationInstance)->onQueue('sales-channels');
            });
        } while ($productsToCreate->count() == $count);

        SphinxSearch::rebuild('products');

        // cache quantity for product listings
        GenerateCacheProductListingQuantityJob::dispatchSync($this->integrationInstance);
        // cache price for product listings
        CacheProductListingPriceJob::dispatchSync($this->integrationInstance);
    }

    private function insertProductsDataToTemporaryTable(array $productsData)
    {
        // create the temporary table
        DB::statement($this->createTemporaryTableQuery());
        // DO NOT use truncate(or drop table), it causes a transaction exception
        DB::delete("DELETE FROM {$this->temporaryProductsTable}");
        // insert data to the temporary table
        $columns = $this->columnize(array_keys(reset($productsData)));
        $values = $this->parameterize($productsData);
        DB::insert("INSERT IGNORE INTO {$this->temporaryProductsTable} ($columns) VALUES $values");
    }

    private function createTemporaryTableQuery(): string
    {
        $customDatatype = ['attributes.description' => 'text'];
        $mappingFields = '';
        foreach ($this->mappings as $mapping) {
            $type = 'varchar(255)';
            if (Str::startsWith($mapping['listing_field'], 'attributes.')) {
                $attributeCode = str_replace('attributes.', '', $mapping['listing_field']);
                $magentoProductAttribute = ProductAttribute::query()->firstWhere('code', $attributeCode);
                if ($magentoProductAttribute && $magentoProductAttribute->type == 'text') {
                    $type = 'text';
                }
            }
            $column = $this->temporaryColumnName($mapping['listing_field']);
            $mappingFields .= "`$column` $type,";
        }
        $uniqueColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'sku')['listing_field']);

        $collate = config('database.connections.mysql.collation');

        return <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS {$this->temporaryProductsTable} (
                `magento_product_id` bigint(20) unsigned, 
                `variant_id` bigint(20) unsigned, 
                `product_id` bigint(20) unsigned,
                `magento_product_title` varchar(255),
                `magento_product_sku` varchar(255),
                `magento_product_price` decimal(9,4),
                `magento_type_id` varchar(255),
                `product_exists` tinyint(1) default 0, 
                `supplier_id` bigint(20) unsigned,
                `supplier_product_id` bigint(20) unsigned, 
                `supplier_warehouse_id` bigint(20) unsigned, 
                `product_listing_id` bigint(20) unsigned,
                `brand_id` bigint(20) unsigned,
                `category_id` bigint(20) unsigned,
                $mappingFields
                PRIMARY KEY (`magento_product_id`), INDEX(`variant_id`), UNIQUE (`$uniqueColumn`), INDEX (`product_id`), INDEX (`product_exists`), INDEX (`supplier_id`), INDEX(`supplier_id`, `product_id`), INDEX(`supplier_product_id`), INDEX(`supplier_warehouse_id`), INDEX(`product_listing_id`), INDEX (`category_id`)
            ) ENGINE=InnoDB DEFAULT COLLATE={$collate}
        SQL;
    }

    private function insertProductsQuery(): string
    {
        $productColumns = $this->mappings->whereIn('sku_field', (new \App\Models\Product())->getFillable());
        // add brand_id
        if ($this->mappings->firstWhere('sku_field', 'brand')) {
            $productColumns->push(['listing_field' => 'brand_id', 'sku_field' => 'brand_id']);
        }
        $notNullColumns = ['weight' => 0, 'length' => 0, 'width' => 0, 'height' => 0];
        $listingColumns = $productColumns->map(function ($c) use ($notNullColumns) {
            $listingColumn = "`tp`.`{$this->temporaryColumnName($c['listing_field'])}`";

            if (in_array($c['sku_field'], array_keys($notNullColumns))) {
                $listingColumn = "IFNULL({$listingColumn}, {$notNullColumns[$c['sku_field']]})";
            }

            return $listingColumn;
        });

        return <<<SQL
                INSERT INTO products (`type`, {$productColumns->pluck('sku_field')->implode(',')}, `created_at`, `updated_at`) 
                SELECT IF(`tp`.`magento_type_id` = 'simple', 'standard', 'bundle') type, {$listingColumns->implode(',')}, NOW(), NOW() FROM {$this->temporaryProductsTable} tp WHERE product_id IS NULL;
                SQL;
    }

    /**
     * Add bundle components to bundle products
     */
    private function addBundleComponents()
    {
        $temporaryBundleComponentsTable = 'temporary_bundle_components';
        $collate = config('database.connections.mysql.collation');
        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE $temporaryBundleComponentsTable (
                `magento_product_id` bigint(20) unsigned,
                `product_id` bigint(20) unsigned,
                `product_listing_id` bigint(20) unsigned,
                `component_sku` varchar(255),
                `component_quantity` int unsigned,
                `compnent_product_id` bigint(20) unsigned,
                PRIMARY KEY (`magento_product_id`, `component_sku`)
            ) ENGINE=InnoDB COLLATE={$collate}
        SQL;

        $bundleProductsQuery = <<<SQL
            SELECT mp.id, tp.product_id, tp.product_listing_id, json_extract(`mp`.`json_object`, '$.extension_attributes.bundle_product_options[*].product_links[0]') bundle_components
            FROM magento_products mp INNER JOIN {$this->temporaryProductsTable} tp 
                ON mp.id = tp.magento_product_id AND tp.magento_type_id = 'bundle' 
            WHERE tp.product_exists = 0
        SQL;

        $bundleComponentsData = [];
        foreach (DB::select($bundleProductsQuery) as $bundleProduct) {
            foreach (json_decode($bundleProduct->bundle_components) as $bundleComponent) {
                $bundleComponentsData[] = "({$bundleProduct->id}, {$bundleProduct->product_id}, {$bundleProduct->product_listing_id}, '$bundleComponent->sku', {$bundleComponent->qty}, NULL)";
            }
        }

        if (empty($bundleComponentsData)) {
            return;
        }

        // add components to the temp table
        DB::statement($createTemporaryTableQuery);
        $values = implode(',', $bundleComponentsData);
        DB::insert("INSERT INTO $temporaryBundleComponentsTable VALUES $values");

        // add component product ids
        $addComponentProductIdsQuery = <<<SQL
            UPDATE $temporaryBundleComponentsTable tbc 
                INNER JOIN magento_products mp ON mp.sku = tbc.component_sku AND mp.integration_instance_id = ? AND mp.product IS NOT NULL 
                INNER JOIN product_listings pl ON mp.product = pl.id 
            SET tbc.compnent_product_id = pl.product_id
        SQL;
        DB::update($addComponentProductIdsQuery, [$this->integrationInstance->id]);

        /** we don't need these deletions because we already excluded bundle products that still have an unmapped component
         * @see filterBundleProductsToCreate
         *
        // delete bundle products that have components still are not mapped/downloaded
        $this->deleteBundleProductsThatMissingComponents($temporaryBundleComponentsTable);
         */

        // add bundle components to products
        $insertBundleComponentsQuery = <<<SQL
            INSERT INTO product_bundle_components (parent_product_id, child_product_id, quantity, created_at, updated_at) 
            SELECT product_id, compnent_product_id, component_quantity, NOW(), NOW() FROM $temporaryBundleComponentsTable
        SQL;
        DB::insert($insertBundleComponentsQuery);
    }

    private function deleteBundleProductsThatMissingComponents(string $temporaryBundleComponentsTable)
    {
        // delete from product_listings
        $deleteUnmappedFromProductListingsQuery = <<<SQL
            DELETE FROM product_listings WHERE id IN (SELECT product_listing_id FROM $temporaryBundleComponentsTable WHERE compnent_product_id IS NULL)
        SQL;
        DB::delete($deleteUnmappedFromProductListingsQuery);

        // delete from products
        $deleteUnmappedFromProductsQuery = <<<SQL
            DELETE FROM products WHERE id IN (SELECT product_id FROM $temporaryBundleComponentsTable WHERE compnent_product_id IS NULL)
        SQL;
        DB::delete($deleteUnmappedFromProductsQuery);

        // delete from the temporary products table
        $deleteUnmappedFromTempProductsQuery = <<<SQL
            DELETE FROM {$this->temporaryProductsTable}
                   WHERE magento_product_id IN (SELECT magento_product_id FROM $temporaryBundleComponentsTable WHERE compnent_product_id IS NULL)
        SQL;
        DB::delete($deleteUnmappedFromTempProductsQuery);

        // delete from the temporary bundle components table
        $deleteUnmappedFromTempBundleQuery = <<<SQL
            DELETE FROM $temporaryBundleComponentsTable
                WHERE magento_product_id IN (SELECT magento_product_id FROM $temporaryBundleComponentsTable WHERE compnent_product_id IS NULL)
        SQL;
        DB::delete($deleteUnmappedFromTempBundleQuery);
    }

    private function addProductImages()
    {
        if ($this->mappings->whereIn('sku_field', ['image', 'image_url', 'other_images'])->isEmpty()) {
            return;
        }

        $primaryImageColumn = $this->mappings->whereIn('sku_field', ['image', 'image_url'])->first()['listing_field'] ?? null;
        if (! $primaryImageColumn) {
            $primaryImageColumn = $this->mappings->firstWhere('sku_field', 'other_images')->first()['listing_field'];
        }
        $primaryImageColumn = $this->temporaryColumnName($primaryImageColumn);

        $unionOtherImagesSQL = '';
        foreach ($this->mappings->where('sku_field', 'other_images') as $otherImage) {
            $otherImageColumn = $this->temporaryColumnName($otherImage['listing_field']);
            // the image is not the primary image
            if ($otherImageColumn != $primaryImageColumn) {
                $unionOtherImagesSQL .= "UNION SELECT product_id, `{$otherImageColumn}`, 0 FROM {$this->temporaryProductsTable} WHERE product_exists = 0\n";
            }
        }

        $query = <<<SQL
            INSERT IGNORE INTO product_images (product_id, url, is_primary, created_at, updated_at) 
            SELECT product_id, url, is_primary, NOW(), NOW() FROM (
                SELECT product_id, `$primaryImageColumn` url, 1 is_primary FROM {$this->temporaryProductsTable} WHERE product_exists = 0 
                {$unionOtherImagesSQL}
                ) images
            WHERE images.url IS NOT NULL
            SQL;

        DB::insert($query);
    }

    private function insertSupplierProductQuery(): string
    {
        $additionalColumns = [
            'default_supplier_sku' => 'supplier_sku',
            'default_supplier_leadtime' => 'leadtime',
            'default_supplier_moq' => 'minimum_order_quantity',
        ];
        $additionalMappedColumns = $this->mappings->whereIn('sku_field', array_keys($additionalColumns))->keyBy('sku_field');

        $additionalMapped = collect($additionalColumns)->only($additionalMappedColumns->keys());
        $supplierProductColumns = $additionalMapped->isNotEmpty() ? (','.$additionalMapped->values()->implode(',')) : '';
        $listingsColumns = $additionalMapped->isNotEmpty() ? (','.$additionalMapped->keys()->map(fn ($k) => "`{$this->temporaryColumnName($additionalMappedColumns[$k]['listing_field'])}`")->implode(',')) : '';

        return <<<SQL
            INSERT INTO supplier_products (product_id, supplier_id, is_default, created_at, updated_at {$supplierProductColumns})
            SELECT product_id, supplier_id, 1, NOW(), NOW() {$listingsColumns} FROM {$this->temporaryProductsTable} WHERE supplier_id IS NOT NULL;
            SQL;
    }

    private function setWarehouseToMappedLinesFromPriorityWarehousesQuery(array $priorityWarehouses): string
    {
        $priorityWarehouses = implode(',', $priorityWarehouses);

        return <<<SQL
            UPDATE sales_order_lines sol2 INNER JOIN (
                SELECT sol.id,
                       pi.warehouse_id,
                       ROW_NUMBER() OVER (PARTITION BY sol.id ORDER BY FIELD(pi.warehouse_id, $priorityWarehouses)) priority
                FROM sales_order_lines sol
                    INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
                    INNER JOIN products_inventory pi ON sol.product_id = pi.product_id AND pi.warehouse_id IN ($priorityWarehouses) AND pi.inventory_available >= sol.quantity
                WHERE sol.warehouse_id IS null) lpw ON lpw.id = sol2.id AND lpw.priority = 1
            SET sol2.warehouse_id = lpw.warehouse_id
            SQL;
    }

    private function setWarehouseToMappedLinesFromSupplierInventoryQuery(): string
    {
        return <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
                INNER JOIN supplier_products sp ON sol.product_id = sp.product_id AND sp.is_default = 1
                INNER JOIN supplier_inventory si ON sol.product_id = si.product_id AND sp.supplier_id = si.supplier_id
                INNER JOIN warehouses w ON w.id = si.warehouse_id AND w.supplier_id IS NOT null AND w.dropship_enabled = 1
            SET sol.warehouse_id = si.warehouse_id
            WHERE sol.warehouse_id IS null AND (si.in_stock = 1 OR si.quantity >= sol.quantity)
            SQL;
    }

    private function setWarehouseToMappedLinesFromFirstPriorityWarehouseQuery(array $priorityWarehouses): string
    {
        $firstPriorityWarehouse = $priorityWarehouses[0];

        return <<<SQL
            UPDATE sales_order_lines sol
                INNER JOIN {$this->temporaryProductsTable} tpi ON tpi.product_listing_id = sol.product_listing_id
            SET sol.warehouse_id = $firstPriorityWarehouse
            WHERE sol.warehouse_id IS null
        SQL;
    }

    private function temporaryColumnName(string $listingField)
    {
        return str_replace('.', '-', $listingField);
    }

    private function addProductBrands(): void
    {
        if (! $brandMapping = $this->mappings->firstWhere('sku_field', 'brand')) {
            return;
        }

        $brandColumn = $this->temporaryColumnName($brandMapping['listing_field']);

        // insert brands into product_brands
        $insertQuery = <<<SQL
            INSERT IGNORE INTO `product_brands` (`name`, `created_at`, `updated_at`) 
            SELECT DISTINCT `{$brandColumn}`, NOW(), NOW() FROM `{$this->temporaryProductsTable}` tp 
            WHERE tp.product_exists = 0
            SQL;
        DB::insert($insertQuery);

        // add inserted brand ids into the temporary table
        $lookupQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} AS tp 
                INNER JOIN product_brands AS pb ON pb.name = `tp`.`{$brandColumn}` 
            SET tp.brand_id = pb.id;
           SQL;
        DB::update($lookupQuery);
    }

    private function insertProducts(): void
    {
        DB::insert($this->insertProductsQuery());

        $skuColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'sku')['listing_field']);
        $lookupQuery = "UPDATE {$this->temporaryProductsTable} tp INNER JOIN products p ON p.sku = tp.{$skuColumn} SET tp.product_id = p.id;";
        DB::statement($lookupQuery);
    }

    private function addDefaultSupplierToProducts(?SupplierPricingTier $supplierPricingTier, ?string $supplierPricingListingField): void
    {
        // default supplier is not mapped
        if (! $defaultSupplierMapping = $this->mappings->firstWhere('sku_field', 'default_supplier')) {
            return;
        }

        if ($this->hasNonExistingSuppliers()) {
            $this->createNonExistingSuppliers();
        }

        $supplierColumn = $this->temporaryColumnName($defaultSupplierMapping['listing_field']);

        // set default_warehouse_id to suppliers (exiting suppliers) if it did not set
        $query = <<<SQL
            UPDATE suppliers s
                INNER JOIN (SELECT DISTINCT `tp`.`{$supplierColumn}` supplier_name FROM $this->temporaryProductsTable tp WHERE tp.product_exists = 0) ts ON ts.supplier_name = s.name
                INNER JOIN warehouses w ON s.id = w.supplier_id 
            SET s.default_warehouse_id = w.id 
            WHERE s.default_warehouse_id IS NULL
        SQL;
        DB::update($query);

        // set supplier_id and supplier_warehouse_id to the temporary table
        $lookupSuppliersQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp INNER JOIN suppliers s ON s.name = `tp`.`{$supplierColumn}` 
            SET tp.supplier_id = s.id, tp.supplier_warehouse_id = s.default_warehouse_id 
            WHERE tp.product_exists = 0;
           SQL;
        DB::update($lookupSuppliersQuery);

        // add supplier products
        DB::insert($this->insertSupplierProductQuery());
        $lookupSupplierProductsQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN supplier_products sp ON sp.supplier_id = tp.supplier_id AND sp.product_id = tp.product_id 
            SET tp.supplier_product_id = sp.id WHERE tp.supplier_id IS NOT NULL;
           SQL;
        DB::update($lookupSupplierProductsQuery);

        // add supplier product pricing
        if ($supplierPricingTier) {
            $insertSupplierProductPricingQuery = <<<SQL
                INSERT INTO supplier_product_pricing (supplier_product_id, supplier_pricing_tier_id, price, created_at, updated_at) 
                SELECT supplier_product_id, {$supplierPricingTier->id}, `{$this->temporaryColumnName($supplierPricingListingField)}`, NOW(), NOW() 
                FROM {$this->temporaryProductsTable} WHERE supplier_product_id IS NOT NULL AND `{$this->temporaryColumnName($supplierPricingListingField)}` IS NOT NULL;
               SQL;
            DB::insert($insertSupplierProductPricingQuery);
        }

        // initialize inventory to supplier products
        $insertSupplierInventoryQuery = <<<SQL
            INSERT INTO supplier_inventory (product_id, supplier_id, warehouse_id, quantity, in_stock, created_at, updated_at) 
            SELECT tp.product_id, tp.supplier_id, tp.supplier_warehouse_id, IF(s.default_stock_level = ?, 0, NULL), IF(s.default_stock_level = ?, 0, 1), NOW(), NOW() FROM {$this->temporaryProductsTable} tp
                INNER JOIN suppliers s ON s.id = tp.supplier_id;
           SQL;
        DB::insert($insertSupplierInventoryQuery, [Supplier::STOCK_LEVEL_ZERO, Supplier::STOCK_LEVEL_NOT_IN_STOCK]);
    }

    protected function hasNonExistingSuppliers(): bool
    {
        $supplierColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'default_supplier')['listing_field']);

        $query = <<<SQL
            SELECT COUNT(DISTINCT `tp`.`{$supplierColumn}`) as missing FROM $this->temporaryProductsTable tp
                    LEFT JOIN suppliers s ON `tp`.`{$supplierColumn}` = s.name 
                WHERE s.name IS NULL AND `tp`.`{$supplierColumn}` IS NOT NULL AND `tp`.`{$supplierColumn}` != '' AND tp.product_exists = 0
        SQL;

        return DB::selectOne($query)->missing > 0;
    }

    protected function createNonExistingSuppliers()
    {
        $supplierColumn = $this->temporaryColumnName($this->mappings->firstWhere('sku_field', 'default_supplier')['listing_field']);

        $tempSuppliersTable = 'temporary_suppliers_table';
        $collate = config('database.connections.mysql.collation');
        $query = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS $tempSuppliersTable 
            (
                `name` VARCHAR(255),
                `supplier_id` BIGINT(20) UNSIGNED,
                UNIQUE(`name`)
            )
            ENGINE=InnoDB COLLATE={$collate}
        SQL;
        DB::statement($query);
        // DO NOT use truncate(or drop table), it causes a transaction exception
        DB::delete("DELETE FROM {$tempSuppliersTable}");

        // insert into the suppliers temporary table
        $query = <<<SQL
            INSERT INTO $tempSuppliersTable (name)
            SELECT DISTINCT `tp`.`{$supplierColumn}` FROM $this->temporaryProductsTable tp
                LEFT JOIN suppliers s ON `tp`.`{$supplierColumn}` = s.name
            WHERE s.name IS NULL AND `tp`.`{$supplierColumn}` IS NOT NULL AND `tp`.`{$supplierColumn}` != '' AND tp.product_exists = 0
        SQL;
        DB::insert($query);

        // Create suppliers
        $query = <<<SQL
            INSERT INTO suppliers (`name`, `created_at`, `updated_at`)
                SELECT `name`, NOW(), NOW() FROM $tempSuppliersTable WHERE `name` IS NOT NULL;
        SQL;
        DB::statement($query);

        // Update supplier ids
        DB::statement("
            UPDATE $tempSuppliersTable ts
                INNER JOIN suppliers s ON s.name = ts.name 
            SET ts.supplier_id = s.id 
        ");

        // Create supplier addresses
        $query = <<<SQL
            INSERT INTO addresses (`name`, `created_at`, `updated_at`)
                SELECT `name`, NOW(), NOW() FROM $tempSuppliersTable
        SQL;
        DB::statement($query);

        // Create supplier warehouses
        $query = <<<SQL
            INSERT INTO warehouses (`name`, `type`, `supplier_id`, `created_at`, `updated_at`) 
            SELECT ?, ?, `supplier_id`, NOW(), NOW() 
            FROM $tempSuppliersTable WHERE `supplier_id` IS NOT NULL
        SQL;
        DB::statement($query, [Supplier::DEFAULT_WAREHOUSE_NAME, Warehouse::TYPE_SUPPLIER]);

        // set warehouses as the default warehouse to suppliers
        $query = <<<SQL
            UPDATE suppliers s INNER JOIN {$tempSuppliersTable} ts ON ts.supplier_id = s.id 
                INNER JOIN warehouses w ON ts.supplier_id = w.supplier_id 
            SET s.default_warehouse_id = w.id
        SQL;
        DB::update($query);
    }

    private function addPricingToProducts(): void
    {
        $pricingMappings = $this->mappings
            ->filter(fn ($m) => preg_match("/price\..*\.value/", $m['sku_field']))
            ->map(fn ($m) => ['name' => explode('.', $m['sku_field'])[1], 'listing_column' => $this->temporaryColumnName($m['listing_field'])])
            ->keyBy('name');

        $supplierPricingTiers = ProductPricingTier::query()->whereIn('name', $pricingMappings->keys())->get();

        // download Magento customer groups
        (new GetCustomerGroupsJob($this->integrationInstance))->handle();

        // prepare pricing data
        $pricingColumns = $pricingMappings->map(fn ($m) => "`tp`.`{$m['listing_column']}`");
        $pricingColumns = $pricingColumns->isEmpty() ? '' : ", {$pricingColumns->implode(', ')}";
        // add product tier prices to a temporary table
        $selectQuery = <<<SQL
            SELECT `mp`.`id`, `tp`.`product_id`, json_extract(`mp`.`json_object`, '$.tier_prices') tier_prices {$pricingColumns}
            FROM {$this->temporaryProductsTable} tp INNER JOIN magento_products mp 
                ON tp.magento_product_id = mp.id AND mp.integration_instance_id = {$this->integrationInstance->id}
            WHERE tp.product_exists = 0
        SQL;
        $pricingData = [];
        foreach (DB::select($selectQuery) as $item) {
            // add the mapped pricing tiers
            foreach ($pricingMappings as $pricingMapping) {
                $supplierPricingTierId = $supplierPricingTiers->firstWhere('name', $pricingMapping['name'])->id;
                if ($value = $item->{$pricingMapping['listing_column']}) {
                    $pricingData[] = "(NULL, {$item->id}, {$item->product_id}, {$supplierPricingTierId}, NULL, {$value})";
                }
            }
            // add the product tier prices
            foreach (json_decode($item->tier_prices) as $tierPrice) {
                if ($tierPrice->qty == 1) {
                    $pricingData[] = "(NULL, {$item->id}, {$item->product_id}, NULL, {$tierPrice->customer_group_id}, {$tierPrice->value})";
                }
            }
        }

        if (empty($pricingData)) {
            return;
        }

        // create a temporary table
        $collate = config('database.connections.mysql.collation');
        $temporaryPricingTable = 'temporary_product_pricing';
        $createTempPricingTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS {$temporaryPricingTable} (
                `id` bigint(20) unsigned auto_increment primary key,
                `magento_product_id` bigint(20) unsigned,
                `product_id` bigint(20) unsigned,
                `product_pricing_tier_id` bigint(20) unsigned,
                `customer_group_id` bigint(20) unsigned,
                `value` float
            ) ENGINE=InnoDB COLLATE={$collate}
        SQL;
        DB::statement($createTempPricingTableQuery);
        // DO NOT use truncate(or drop table), it causes a transaction exception
        DB::delete("DELETE FROM {$temporaryPricingTable}");

        // insert product pricing to the temporary table
        $insertQuery = "INSERT INTO $temporaryPricingTable VALUES ".implode(',', $pricingData);
        DB::insert($insertQuery);

        // add product pricing tiers by customer_group_id
        $query = <<<SQL
            UPDATE {$temporaryPricingTable} ttp 
                INNER JOIN magento_customer_groups mcg 
                ON mcg.customer_group_id = ttp.customer_group_id AND mcg.integration_instance_id = ?
            SET ttp.product_pricing_tier_id = mcg.product_pricing_tier_id
            WHERE ttp.customer_group_id IS NOT NULL
        SQL;
        DB::update($query, [$this->integrationInstance->id]);

        // add pricing tiers to products
        // TODO: what should we do if the selected mapping has the same product pricing tier of the magento tier prices (customer groups)
        $insertPricingTiersQuery = <<<SQL
            INSERT INTO product_pricing (`product_pricing_tier_id`, `product_id`, `price`, `created_at`, `updated_at`)
            SELECT ttp.product_pricing_tier_id, ttp.product_id, ttp.value, NOW(), NOW() 
            FROM {$temporaryPricingTable} ttp
            ON DUPLICATE KEY UPDATE `price` = VALUES(`price`)
        SQL;
        DB::insert($insertPricingTiersQuery);
    }

    private function addAttributesToProducts(): void
    {
        $attributeMappings = $this->mappings
            ->filter(fn ($m) => Str::startsWith($m['sku_field'], 'attributes.'))
            ->map(fn ($m) => [
                'name' => str_replace('attributes.', '', $m['sku_field']),
                'listing_column' => $this->temporaryColumnName($m['listing_field']),
            ])->keyBy('name');

        // there are no mappings to attributes
        if ($attributeMappings->isEmpty()) {
            return;
        }

        Attribute::query()
            ->whereIn('name', $attributeMappings->keys())
            ->each(function (Attribute $attribute) use ($attributeMappings) {
                $listingColumn = $attributeMappings[$attribute->name]['listing_column'];
                $insertProductAttributesQuery = <<<SQL
                        INSERT INTO product_attributes (product_id, attribute_id, `value`, created_at, updated_at) 
                        SELECT product_id, {$attribute->id}, `{$listingColumn}`, NOW(), NOW() FROM {$this->temporaryProductsTable} 
                        WHERE product_exists = 0 AND `{$listingColumn}` IS NOT NULL;
                       SQL;

                DB::insert($insertProductAttributesQuery);
            });
    }

    private function addMainCategoryToProducts(): void
    {
        // category_main is not mapped
        if (! $categoryMapping = $this->mappings->firstWhere('sku_field', 'category_main')) {
            return;
        }

        $listingColumn = $this->temporaryColumnName($categoryMapping['listing_field']);
        $lookupCategoriesQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN product_categories pc ON pc.name = `tp`.`{$listingColumn}` AND pc.parent_id IS NULL 
            SET tp.category_id = pc.id WHERE tp.category_id IS NULL;
           SQL;
        DB::update($lookupCategoriesQuery);

        // insert the non-existent categories
        $insertProductCategoriesQuery = <<<SQL
            INSERT INTO `product_categories` (name, is_leaf, path, level, created_at, updated_at) 
            SELECT `tp`.`{$listingColumn}`, 1, `tp`.`{$listingColumn}`, 1, NOW(), NOW() 
            FROM {$this->temporaryProductsTable} tp 
            WHERE tp.category_id IS NULL AND `tp`.`{$listingColumn}` IS NOT NULL AND product_exists = 0
           SQL;
        DB::insert($insertProductCategoriesQuery);
        DB::update($lookupCategoriesQuery);

        // add categories to products
        $addCategoriesToProductsQuery = <<<SQL
            INSERT INTO product_to_categories (product_id, category_id, is_primary, created_at, updated_at)
            SELECT tp.product_id, tp.category_id, 1, NOW(), NOW() 
            FROM {$this->temporaryProductsTable} tp WHERE tp.category_id IS NOT NULL AND product_exists = 0;
           SQL;
        DB::insert($addCategoriesToProductsQuery);

        // add attributes to products from category's attribute groups
        $addAttributesFromAttributeGroupsQuery = <<<SQL
            INSERT INTO product_attributes (product_id, attribute_id, attribute_group_id, created_at, updated_at) 
            SELECT tp.product_id, att.id, att.attribute_group_id, NOW(), NOW() FROM {$this->temporaryProductsTable} tp 
                INNER JOIN categories_to_attribute_groups ctag ON ctag.category_id = tp.category_id 
                INNER JOIN attributes att ON att.attribute_group_id = ctag.attribute_group_id 
            WHERE tp.category_id IS NOT NULL AND tp.product_exists = 0
            ON DUPLICATE KEY UPDATE `attribute_group_id` = VALUES(`attribute_group_id`)
           SQL;
        DB::insert($addAttributesFromAttributeGroupsQuery);

        // add category's attributes to products
        $addCategoryAttributesToProducts = <<<SQL
            INSERT IGNORE INTO product_attributes (product_id, attribute_id, created_at, updated_at) 
            SELECT tp.product_id, cta.attribute_id, NOW(), NOW() FROM {$this->temporaryProductsTable} tp 
                INNER JOIN categories_to_attributes cta ON cta.category_id = tp.category_id 
            WHERE tp.category_id IS NOT NULL AND tp.product_exists = 0;
           SQL;
        DB::insert($addCategoryAttributesToProducts);
    }

    private function insertProductListings(): void
    {
        $InsertProductListingsQuery = <<<SQL
            INSERT INTO product_listings (`product_id`, `sales_channel_id`, `sales_channel_listing_id`, `listing_sku`, `title`, `price`, `quantity`, `document_id`, `created_at`, `updated_at`, `document_type`) 
            SELECT product_id, ?, variant_id, magento_product_sku, magento_product_title, magento_product_price, 0, magento_product_id, NOW(), NOW(), ?
            FROM {$this->temporaryProductsTable}
            ON DUPLICATE KEY UPDATE `document_id` = VALUES(`document_id`), `updated_at` = VALUES(`updated_at`), `document_type` = VALUES(`document_type`);
        SQL;
        DB::insert($InsertProductListingsQuery, [$this->integrationInstance->salesChannel->id, (new Product())->getMorphClass()]);

        $lookupProductListingsQuery = <<<SQL
            UPDATE {$this->temporaryProductsTable} tp 
                INNER JOIN product_listings pl ON pl.sales_channel_listing_id = tp.variant_id AND pl.sales_channel_id = ? 
                INNER JOIN magento_products mp ON mp.id = tp.magento_product_id 
            SET tp.product_listing_id = pl.id, mp.product = pl.id, mp.mapped_at = NOW();
           SQL;
        DB::update($lookupProductListingsQuery, [$this->integrationInstance->salesChannel->id]);
    }

    /**
     * filter bundle components that have an unmapped component
     */
    private function filterBundleProductsToCreate(array $productsToInsert): array
    {
        $componentSkus = collect($productsToInsert)->pluck('component_skus')->flatten()->unique();
        if ($componentSkus->isEmpty()) {
            return $productsToInsert;
        }
        $products = collect($productsToInsert)->keyBy('magento_product_sku');
        $components = Product::query()->whereIn('sku', $componentSkus)->select(['sku', 'status', 'product'])->get()->keyBy('sku');

        return array_filter(array_map(function ($p) use ($components, $products) {
            if ($p['magento_type_id'] != 'bundle') {
                return $p;
            }

            foreach ($p['component_skus'] as $componentSku) {
                // still is not downloaded(or removed from Magento)
                if (! isset($components[$componentSku]) ||
                   // unmapped and not in products that need to create
                   (! $components[$componentSku]->product && ! isset($products[$componentSku]))) {
                    return null;
                }
            }

            // bundle product and all components are enabled
            unset($p['component_skus']);

            return $p;
        }, $productsToInsert));
    }

    private function columnize($columns): string
    {
        return implode(', ', array_map(fn ($k) => "`$k`", $columns));
    }

    private function parameterize($data): string
    {
        return collect($data)
            ->map(function ($record) {
                $values = array_map(function ($v) {
                    return is_null($v) ? 'null' : ("'".str_replace("'", "\'", stripslashes($v))."'");
                }, array_values($record));

                return '('.implode(',', $values).')';
            })
            ->implode(', ');
    }
}
