<?php

namespace App\Jobs;

use App\Exporters\Jasper\PackingSlipTransformer;
use App\Ghostscript\Ghostscript;
use App\Models\PackingSlipQueue;
use App\Models\SalesOrderFulfillment;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;

class GenerateSalesOrderFulfillmentPackingSlipsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, JasperExport, Queueable, SerializesModels;

    private $oneFile = false;

    private $markAsPrinted = false;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected array|int $salesOrderFulfillmentIds, protected ?PackingSlipQueue $packingSlipQueue = null, protected array $options = [])
    {
        $this->salesOrderFulfillmentIds = Arr::wrap($salesOrderFulfillmentIds);
        $this->oneFile = ! empty($options['one_file']) && count($this->salesOrderFulfillmentIds) > 1;
        $this->markAsPrinted = ! empty($options['mark_as_printed']);

        if (empty($salesOrderFulfillmentIds)) {
            throw new \InvalidArgumentException('Empty sales order fulfillment ids');
        }
    }

    /**
     * Execute the job.
     *
     * @throws \PHPJasper\Exception\ErrorCommandExecutable
     * @throws \PHPJasper\Exception\InvalidCommandExecutable
     * @throws \PHPJasper\Exception\InvalidFormat
     * @throws \PHPJasper\Exception\InvalidInputFile
     * @throws \PHPJasper\Exception\InvalidResourceDirectory
     * @throws Exception
     */
    public function handle(): ?string
    {
        $this->packingSlipQueue?->printing();

        /** @var Collection|SalesOrderFulfillment[] $salesOrderFulfillments */
        $salesOrderFulfillments = SalesOrderFulfillment::with([
            'salesOrder.store.address',
            'salesOrder.customer.address',
            'salesOrder.notes',
            'salesOrder.salesOrderLines.product.productAttributes',
            'salesOrder.salesOrderLines.product.primaryImage',
            'fulfilledShippingMethod',
            'requestedShippingMethod',
        ])->findMany($this->salesOrderFulfillmentIds);

        // trying to generate the packing slip after deleting the fulfillment
        if ($salesOrderFulfillments->isEmpty()) {
            // mark packing slip queue as printed (delete it)
            $this->packingSlipQueue?->printed();

            return null;
        }

        $allLineItems = [];
        $generatedFiles = [];
        foreach ($salesOrderFulfillments->groupBy('salesOrder.active_store.custom_jrxml_file') as $templateFile => $salesOrderFulfillments) {
            foreach ($salesOrderFulfillments as $salesOrderFulfillment) {
                if (is_null($salesOrderFulfillment->salesOrder?->customer)) {
                    continue;
                }

                $fulfillmentItems = [];
                $fulfillmentOrderData = PackingSlipTransformer::salesOrderFulfillment($salesOrderFulfillment);

                foreach ($salesOrderFulfillment->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                    foreach (PackingSlipTransformer::salesOrderFulfillmentLine($salesOrderFulfillmentLine) as $line) {
                        $fulfillmentItems[] = array_merge($line, $fulfillmentOrderData);
                    }
                }

                if ($this->oneFile) {
                    $allLineItems[$templateFile] = array_merge($allLineItems[$templateFile] ?? [], $fulfillmentItems);
                } else {
                    $fulfillmentPSFile = Storage::disk('fulfillment_packing_slips')->path($salesOrderFulfillment->id);
                    $generatedFiles[] = $this->export($fulfillmentItems, $this->getTemplatePath($templateFile), $fulfillmentPSFile);
                }
            }
        }

        if ($this->oneFile) {
            foreach ($allLineItems as $templateFile => $items) {
                $orderPSFile = Storage::disk('fulfillment_packing_slips')->path(time().rand(1000, 9999));
                $generatedFiles[] = $this->export($items, $this->getTemplatePath($templateFile), $orderPSFile);
            }
        }

        //Update packing
        /*
         * Timestamp should only be touched here if the slip printing
         * was requested by the user.
         */
        if ($this->markAsPrinted) {
            SalesOrderFulfillment::whereIn('id', $this->salesOrderFulfillmentIds)->update([
                'packing_slip_printed_at' => now(),
            ]);
        }

        // mark packing slip queue as printed (delete it)
        $this->packingSlipQueue?->printed();

        if (empty($generatedFiles)) {
            return null;
        }

        // if only one file, return it
        if (count($generatedFiles) == 1) {
            return $generatedFiles[0];
        }

        // if more than one file, combine them in one file and return it
        $outputFile = Storage::disk('fulfillment_packing_slips')->path(time().rand(1000, 9999).'.pdf');
        if (is_null($combineStatus = (new Ghostscript($outputFile, $generatedFiles))->combine())) {
            return $outputFile;
        }

        throw new Exception("Ghostscript error: {$combineStatus}");
    }

    public function failed(\Throwable $exception)
    {
        $this->packingSlipQueue->failed($exception->getMessage());
    }

    protected function defaultJasperTemplateFile(): string
    {
        return 'SKU_Packing_Slip.jrxml';
    }

    public function tags(): array
    {
        $tags = ['ids:'.implode(',', $this->salesOrderFulfillmentIds)];
        if ($this->packingSlipQueue) {
            $tags[] = "PackingSlipQueue:{$this->packingSlipQueue->id}";
        }

        return $tags;
    }
}
