<?php

namespace App\Jobs;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Models\IntegrationInstance;
use App\Repositories\ProductRepository;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Bus;
use Throwable;

class GenerateCacheProductListingQuantityJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public ProductRepository $products;

    public function __construct(
        protected IntegrationInstance|IntegrationInstanceInterface $integrationInstance,
        protected array $productIds = []
    )
    {
        $this->onQueue('syncInventory');
        $this->products = app(ProductRepository::class);
    }

    /**
     * @throws Throwable
     */
    public function handle(): void
    {
        if (!$this->integrationInstance->salesChannel) {
            return;
        }

        if (empty($this->productIds)) {
            $this->productIds = $this->products->getProductIdsNeedingListingCacheUpdate($this->integrationInstance);
        }

        $jobs = [];
        foreach (array_chunk($this->productIds, 1000) as $chunk) {
            $jobs[] = new CacheProductListingQuantityJob($this->integrationInstance, $chunk);
        }

        if (!empty($jobs)) {
            Bus::batch($jobs)->name('Cache Product Listings Quantity')->dispatch();
        }
    }
}
