<?php

namespace App\Jobs;

use App\Ghostscript\Ghostscript;
use App\Models\PackingSlipQueue;
use App\Models\SalesOrder;
use App\Models\User;
use App\Notifications\SendSalesOrderPackingSlipNotification;
use App\Observers\AddPackingSlipQueueObserver;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

class ExportSalesOrderPackingSlipJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(private User $user, private mixed $filters)
    {
    }

    public function handle(): void
    {
        $exists        = [];
        $needsPrinting = [];
        $count         = 0;

        SalesOrder::filter(['filters' => $this->filters])
            ->select('id')
            ->chunk(50, function ($salesOrders) use (&$exists, &$needsPrinting, &$count) {
                $count += count($salesOrders);
                $ids    = $salesOrders->pluck('id')->toArray();
                $exists = [];
                foreach ($ids as $id) {
                    if (Storage::disk('order_packing_slips')->exists("$id.pdf")) {
                        $exists[] = Storage::disk('order_packing_slips')->path("$id.pdf");
                    } // the file does not exist
                    else  if (!Storage::disk('order_packing_slips')->exists("$id.pdf")) {
                        $needsPrinting[] = $id;
                        AddPackingSlipQueueObserver::addPackingSlipQueue(
                            (new SalesOrder())->forceFill(['id' => (int) $id]), false
                        );
                    } // generating the packing slip still in-progress (in the queue)
                    elseif (PackingSlipQueue::firstWhere(['link_id' => (int) $id, 'link_type' => SalesOrder::class])) {
                        $needsPrinting[] = $id;
                    }
                }
            });

        if (count($needsPrinting)) {
            $generatingPackingSlipResponse = (new GenerateSalesOrderPackingSlipsJob(
                $needsPrinting,
                null, ['one_file' => true]
            ))->handle();

            $errors = $generatingPackingSlipResponse['errors'] ?? [];

            SalesOrder::whereIn('id', array_diff($needsPrinting, array_keys($errors)))
                ->update(['packing_slip_printed_at' => now()]);

            $outputFile    = time().rand(1000, 9999).'.pdf';
            $combineStatus = (new Ghostscript(Storage::disk('reports')
                ->path($outputFile), array_merge([$generatingPackingSlipResponse['generatedFile']], $exists)))
                ->combine();

            if (!is_null($combineStatus)) {
                throw new \Exception('Error in generating file');
            }
            $file = Storage::disk('reports')->url($outputFile);

            $this->user->notify(new SendSalesOrderPackingSlipNotification($count, url($file)));
        }
    }
}
