<?php

namespace App\Jobs;

use App\DataTable\DataTableResource;
use App\DataTable\Exports\DataTableExporter;
use App\Exporters\TransformsExportData;
use App\Models\Concerns\HasFilters;
use App\Models\User;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ExportDataTable implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public string $model;

    public string $resource;

    public array $included = [];

    public string $resourceFileName;

    public string $format;

    public array $mapping = [];

    public string $exportFile;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(public User $user, public array $ids)
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Remove maximum execution time
        set_time_limit(0);
        ini_set('memory_limit', '1024M');

        if ($this->batch()->cancelled()) {
            return;
        }

        $this->process();
    }

    public function makeExporter(array $records): DataTableExporter
    {
        return new DataTableExporter(
            $records,
            $this->resourceFileName,
            $this->format,
            $this->user->id,
            $this->mapping,
            $this->included
        );
    }

    protected function process()
    {
        $records = $this->fetchData();

        $handle = fopen($this->exportFile, 'a');
        foreach ($records as $record) {
            $exporter = $this->makeExporter([$record]);
            fputcsv($handle, $exporter->getRecords()[0]);
        }
        fclose($handle);
    }

    protected function fetchData(): array
    {
        $builder = self::buildExportQuery($this->model, $this->ids, $this->included);

        return self::parseResults(
            $builder->get(),
            $this->resource,
            $this->model,
            $this->included
        );
    }

    public static function parseResults($records, string $resource, string $model, array $included = []): array
    {
        DataTableResource::$includeOverride = $included;
        $records = $resource::collection($records);
        DataTableResource::$includeOverride = null;

        $parsedRecords = json_decode($records->toJson(), true);

        if (in_array(TransformsExportData::class, class_implements($model))) {
            $parsedRecords = $model::transformExportData($parsedRecords);
        }

        return $parsedRecords;
    }

    public static function buildExportQuery(string $model, array $ids = [], ?array $included = [])
    {
        $builder = $model::with([]);

        if (empty($ids)) {
            return $builder;
        }

        if ($builder instanceof Model ||
            $builder instanceof Builder) {
            $tableName = null;
            if ($builder instanceof Builder) {
                $tableName = $builder->getModel()->getTable();
            } elseif ($builder instanceof Model) {
                $tableName = $builder->getTable();
            }
            $column = $tableName ? $tableName.'.id' : 'id';
            $builder = $builder->whereIn($column, $ids);
        } else {
            /**
             * We order the records based on the original
             * ordering based on the ids.
             */
            $idsString = implode(',', $ids);
            $builder = $builder->whereIn('id', $ids)
                ->orderByRaw("FIELD(`id`, $idsString)");
        }

        $usedTraits = class_uses_recursive($builder->getModel());
        if (in_array(HasFilters::class, $usedTraits)) {
            $builder->addRelations(['included' => $included]);
            $builder->addCustomColumns();
        }

        return $builder;
    }

    public function model(string $model): self
    {
        $this->model = $model;

        return $this;
    }

    public function resource(string $resource): self
    {
        $this->resource = $resource;

        return $this;
    }

    public function included(array $included): self
    {
        $this->included = $included;

        return $this;
    }

    public function resourceFileName(string $resourceFileName): self
    {
        $this->resourceFileName = $resourceFileName;

        return $this;
    }

    public function format(string $format): self
    {
        $this->format = $format;

        return $this;
    }

    public function mapping(array $mapping): self
    {
        $this->mapping = $mapping;

        return $this;
    }

    public function exportFile(string $exportFile): self
    {
        $this->exportFile = $exportFile;

        return $this;
    }
}
