<?php

namespace App\Jobs;

use App\DTO\DataImporterDto;
use App\Exceptions\DataImportNoRecordsImportedException;
use App\Exceptions\DataImportTooLargeException;
use App\Exceptions\DataImportValidationErrorsException;
use App\Exceptions\ImportFailedException;
use App\Importers\ImportableInterface;
use App\Mail\ImportCompletedMail;
use App\Managers\DataImportManager;
use App\Models\User;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class DataImportJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(protected User $user, protected string $modelClass, protected DataImporterDto $dataImporterDto)
    {
        $this->dataImporterDto->user = $user;
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        // Remove maximum execution time
        set_time_limit(0);

        $model = app($this->modelClass);

        $importManager = new DataImportManager($model, $this->dataImporterDto);

        $sendsImportEmail = $importManager->sendsImportEmail();

        if ($sendsImportEmail) {
            $importCompleteMail = new ImportCompletedMail($this->user, $this->modelClass);
        }

        try {
            $importManager->import(true);

            if ($sendsImportEmail) {
                // Email the user about the import
                Mail::to($this->user)->queue($importCompleteMail);
            }
        } catch (DataImportNoRecordsImportedException|DataImportTooLargeException $e) {
            throw new Exception('Unexpected data import error (please contact admin): '.$e->getMessage());
        } catch (DataImportValidationErrorsException|ImportFailedException $e) {
            if ($sendsImportEmail) {
                $errors = get_class($e) === DataImportValidationErrorsException::class ? $e->getValidationErrors() : [
                    'message' => $e->getMessage(),
                ];
                $importCompleteMail->subject('Data Import Completed with Errors');
                $importCompleteMail->markdown('mail.imports.completed_with_errors');
                $fileName = pathinfo($this->dataImporterDto['stored_name'], PATHINFO_FILENAME).'_errors.txt';
                Storage::disk('importing_errors')->put($fileName, json_encode($errors));
                $importCompleteMail->attach(Storage::disk('importing_errors')->path($fileName));
            }
        } finally {
            if ($sendsImportEmail && isset($fileName)) {
                // Cleanup if necessary
                Storage::disk('importing_errors')->delete($fileName);
            }
        }
    }
}
