<?php

namespace App\Jobs;

use App\Integrations\Listing;
use App\Jobs\Shopify\ShopifyCreateSkuProducts;
use App\Models\IntegrationInstance;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

/**
 * Class CreateProductFromListing.
 */
class CreateProductsFromListing implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var array|null
     */
    private $ids = [];

    /**
     * @var array|null
     */
    private $filters = [];

    /**
     * @var array
     */
    private $mappings;

    /**
     * @var IntegrationInstance
     */
    private $instance;

    /**
     * CreateProductFromListing constructor.
     */
    private function __construct(IntegrationInstance $instance, array $mappings = [])
    {
        $this->mappings = $mappings;
        $this->instance = $instance;
    }

    public static function withFilters(array $filters, IntegrationInstance $instance, array $mappings = []): static
    {
        $job = new static($instance, $mappings);
        $job->filters = $filters;
        $job->ids = null;

        return $job;
    }

    public static function withIds(array $ids, IntegrationInstance $instance, array $mappings = []): static
    {
        $job = new static($instance, $mappings);
        $job->ids = array_unique($ids);
        $job->filters = null;

        return $job;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        set_time_limit(0);

        $query = $this->getQuery();

        if ($this->instance->isShopify()) {
            dispatch(new ShopifyCreateSkuProducts($this->instance, $query->pluck('id')->toArray()));

            return;
        }

        $query->chunk(1000, function (Collection $listings) {
            foreach ($listings as $listing) {
                try {
                    /** @var Listing $listing */
                    $listing->createSkuProduct($this->mappings);
                } catch (\Exception $e) {
                    //Log::debug($e->getMessage());
                }
            }
        });
    }

    private function getQuery()
    {
        /** @var Model $productsPath */
        $productsPath = $this->instance->integration->getProductsModelPath();

        if ($this->filters && ! empty($this->filters)) {
            return $productsPath::with(['productListing'])
                ->where('integration_instance_id', $this->instance->id)
                ->filter(['filters' => $this->filters]);
        } else {
            return $productsPath::with(['productListing'])
                ->where('integration_instance_id', $this->instance->id)
                ->whereIn('_id', $this->ids);
        }
    }
}
