<?php

namespace App\Integrations;

use App\Models\IntegrationInstance;
use App\Models\IntegrationInstanceServiceDisruptions;
use App\Models\SalesChannelType;
use App\SDKs\Magento\MagentoSDK;
use App\SDKs\Magento\OauthClient;
use Exception;
use Generator;
use Illuminate\Support\Str;
use InvalidArgumentException;
use OAuth\Common\Consumer\Credentials;

class Magento extends Channel implements HasOAuth
{
    private $magentoSDK;

    /**
     * Magento constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $afterRetrievingCallback = function (string $response, $statusCode) use ($integrationInstance) {
            // to mark integration instance credentials as Invalid
            if ($statusCode == 401 || ($statusCode == 400 && Str::contains($response, 'The token length is invalid'))) {
                $integrationInstance->unauthorizedConnection();
            }
            // set connection status for the integration instance
            if (Str::contains($statusCode, ['Operation timed out', 'Could not resolve host', 'matches target host'])) {
                $integrationInstance->setConnectionStatus(IntegrationInstanceServiceDisruptions::CONNECTION_STATUS_DOWN);
            } else {
                $integrationInstance->setConnectionStatus(IntegrationInstanceServiceDisruptions::CONNECTION_STATUS_UP);
            }
        };

        $this->magentoSDK = new MagentoSDK($integrationInstance->credentials, $afterRetrievingCallback);
    }

    /**
     * Get Orders from sales channel.
     */
    public function getSalesOrders($options = null): Generator
    {
        // For initials options.
        if (is_null($options)) {
            $options = [];
        }
        $options['currentPage'] = $options['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getOrders($options);

            $options['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while (count($response['items']) == $response['search_criteria']['page_size']);
    }

    /**
     * Get Products from sales channel.
     */
    public function getProducts($options = null): Generator
    {
        // For initials options.
        if (is_null($options)) {
            $options = [];
        }

        $options['currentPage'] = $options['currentPage'] ?? 1;

        do {
            $response = $this->magentoSDK->getProducts($options);

            $options['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page

            yield [$response, $options];
        } while ($this->hasNextPage($response));
    }

    /**
     * Get order shipments
     */
    public function getShipments($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getShipments($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get order credit memos
     */
    public function getCreditMemos($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getCreditMemos($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get order returns (RMA)
     */
    public function getReturns($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getReturns($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get order invoices
     */
    public function getInvoices($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getInvoices($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get Customer Groups
     */
    public function getCustomerGroups($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getCustomerGroups($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get Stores
     *
     * @param $searchCriteria
     */
    public function getStores(): Generator
    {
        $views = $this->magentoSDK->getStoresViews([]);
        $groups = collect($this->magentoSDK->getStoresGroups([]));
        $websites = collect($this->magentoSDK->getStoresWebsites([]));

        $stores = [];

        foreach ($views as $view) {
            $stores[] = [
                'id' => $view['id'],
                'code' => $view['code'],
                'view_name' => $view['name'],
                'store_name' => $groups->where('id', $view['store_group_id'])->first()['name'],
                'website_name' => $websites->where('id', $view['website_id'])->first()['name'],
                'json_object' => ['view' => $view, 'group' => $groups->where('id', $view['store_group_id'])->first(), 'website' => $websites->where('id', $view['website_id'])->first()],
            ];
        }

        return $stores;
    }

    /**
     * Get Product Attributes
     */
    public function getProductAttributes($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getProductAttributes($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Creates new Shipment for the given Order.
     */
    public function shipOrder(int $orderId, array $shipment): int
    {
        return $this->magentoSDK->shipOrder($orderId, $shipment);
    }

    /**
     * Update base price of products
     */
    public function updateProductBasePrices(array $prices): array
    {
        return $this->magentoSDK->updateProductBasePrices($prices);
    }

    /**
     * Get base price of products
     */
    public function getProductBasePrices($skus): array
    {
        return $this->magentoSDK->getProductBasePrices($skus);
    }

    /**
     * Add or update product prices.
     */
    public function updateProductTierPrices($prices): array
    {
        return $this->magentoSDK->updateProductTierPrices($prices);
    }

    /**
     * Get Inventory Source
     */
    public function getInventorySources($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getInventorySources($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Get Source Items
     */
    public function getSourceItems($searchCriteria = null): Generator
    {
        // For initials options.
        if (is_null($searchCriteria)) {
            $searchCriteria = [];
        }
        $searchCriteria['currentPage'] = $searchCriteria['currentPage'] ?? 1;

        do {
            yield $response = $this->magentoSDK->getSourceItems($searchCriteria);

            $searchCriteria['currentPage'] = $response['search_criteria']['current_page'] + 1; // Next page
        } while ($this->hasNextPage($response));
    }

    /**
     * Update Source Items
     *
     *
     * @return mixed
     */
    public function updateSourceItems(array $sourceItems)
    {
        if (empty($sourceItems)) {
            return null;
        }

        return $this->magentoSDK->updateSourceItems($sourceItems);
    }

    /**
     * Bulk update products
     */
    public function bulkUpdateProducts(array $products): mixed
    {
        return $this->magentoSDK->bulkOperation('products/bySku', 'PUT', $products);
    }

    /**
     * Update a product
     */
    public function updateProduct(string $sku, array $product): array
    {
        return $this->magentoSDK->updateProduct($sku, $product);
    }

    /*
     * Detect if the response has a next page
     *
     * @param array $response
     *
     * @return bool
     */
    private function hasNextPage(array $response): bool
    {
        $itemsCount = count($response['items']);
        // empty items
        if ($itemsCount == 0) {
            return false;
        }
        // items count equals total count
        if ($itemsCount == $response['total_count']) {
            return false;
        }
        // items count less than the selected page size
        if (isset($response['search_criteria']['page_size']) && $itemsCount < $response['search_criteria']['page_size']) {
            return false;
        }

        return true;
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        // TODO: Implement checkCredentials() method.
    }

    /**
     * Generate OAuth user url.
     */
    public function generateOAuthUrl(array $options = []): array
    {
        $state = uniqid(SalesChannelType::TYPE_Magento.'_');
        /**
         * Must add state to urls.
         */
        $callback_url = url(config('sales_channels_oauth.magento.callback_url'), [], true).'?'.http_build_query(['state' => $state]);

        /**
         * Must add "&" to this link to read magento get params correctly.
         */
        $identity_link_url = url(config('sales_channels_oauth.magento.identity_link_url'), [], true)
                         .'?'.http_build_query(['state' => $state]).'&';

        return ['state' => $state, 'callback_url' => $callback_url, 'identity_link_url' => $identity_link_url];
    }

    /**
     * Get user access token from credentials.
     */
    public static function getMagentoAccessToken(array $credentials): array
    {
        /**
         * Check required data.
         */
        if (array_diff(
            [
                'oauth_consumer_key',
                'oauth_consumer_secret',
                'store_base_url',
                'oauth_verifier',
            ],
            array_keys($credentials)
        )) {
            throw new InvalidArgumentException('oauth_consumer_key, oauth_consumer_secret, store_base_url and oauth_verifier are required.');
        }

        /**
         * OAuth Credentials.
         */
        $oauth_credentials = new Credentials(
            $credentials['oauth_consumer_key'],
            $credentials['oauth_consumer_secret'],
            rtrim($credentials['store_base_url'], '/')
        );
        /**
         * Magento OAuth client.
         */
        $oAuthClient = new OauthClient($oauth_credentials);

        try {
            // Get a Request token.
            $request_token = $oAuthClient->requestRequestToken();
            // Get an Access token.
            $access_token = $oAuthClient->requestAccessToken(
                $request_token->getRequestToken(),
                $credentials['oauth_verifier'],
                $request_token->getRequestTokenSecret()
            );
        } catch (Exception $exception) {
            return ['status' => false, 'message' => $exception->getMessage()];
        }

        return [
            'status' => true,
            'access_token' => $access_token->getAccessToken(),
            'secret_token' => $access_token->getAccessTokenSecret(),
        ];
    }

    /**
     * Refresh user access token.
     *
     *
     * @return mixed
     */
    public function refreshAccessToken(string $refresh_token)
    {
        // TODO: Implement refreshAccessToken() method.
    }

    /**
     * Get user access token from code.
     *
     *
     * @return mixed
     */
    public function getAccessToken(string $code = '')
    {
        // TODO: Implement getAccessToken() method.
    }
}
