<?php

namespace App\Integrations\Listings;

use App\Importers\DataImporters\ProductDataImporter;
use App\Importers\Parsers\FieldParserFactory;
use App\Models\Product as SkuProduct;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Str;

trait ListingsHelper
{
    /**
     * @return false|mixed|string|string[]
     *
     * @throws \Exception
     */
    public function parseRules($value, array $parsers)
    {
        if (empty($parsers)) {
            return $value;
        }

        foreach ($parsers as $parser) {
            $value = FieldParserFactory::make($parser['rule'], $parser['args'], $this)->parse($value);
        }

        return $value;
    }

    public function getMappedField($field, array $mapper = []): ?string
    {
        $matched = collect($mapper)->firstWhere('listing_field', $field);

        if (! $matched) {
            return null;
        }

        return $matched['sku_field'];
    }

    /**
     * @throws BindingResolutionException
     * @throws FileNotFoundException
     */
    public function createSkuProductWithMapping(array $fieldMapping, array $overrides = []): ?SkuProduct
    {
        // Create the needed payload from the mapping and
        // create the product.
        $payload = [];
        foreach ($fieldMapping as $mapping) {
            $value = $this->parseRules($this->getValue($mapping['listing_field']), $mapping['parsers'] ?? []);

            if ($mapping['sku_field'] === 'other_images') {
                $payload['other_images'][] = $value;
            } else {
                $payload[$mapping['sku_field']] = $value;
            }
        }

        if (isset($payload['other_images'])) {
            $payload['other_images'] = implode(',', $payload['other_images']);
        }

        // Change tags separator from , to |
        if (! empty($payload['tags'])) {
            $payload['tags'] = str_replace(',', '|', $payload['tags']);
        }

        return ProductDataImporter::createProduct(array_merge($overrides, $payload));
    }

    /**
     * Override this method if values are levels deep.
     *
     *
     * @return mixed
     */
    public function getValue($field)
    {
        // Add in image sources
        if (Str::startsWith($field, 'image')) {
            $images = $this['images'];
            if (empty($images)) {
                return '';
            }
            $imageIndex = max(0, (int) str_replace('image', '', $field) - 1);
            if (is_numeric($imageIndex) && isset($images[$imageIndex])) {
                return $images[$imageIndex]['src'] ?? '';
            }

            return null;
        }

        return $this[$field];
    }
}
