<?php

namespace App\Integrations\Http;

use App\Data\ApiLogData;
use App\Enums\HttpMethodEnum;
use App\Integrations\ApiLogger;
use App\Jobs\UpdateApiLogJob;
use App\Models\IntegrationInstance;
use Exception;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use InvalidArgumentException;
use RuntimeException;
use Throwable;

class IntegrationHttpClient
{

    protected ?string $currentLogUuid = null;

    /**
     * @var ApiLogger Logger
     */
    protected ApiLogger $logger;

    /** @var int Timeout in seconds */
    protected int $timeout = 60;

    /**
     * @param  IntegrationInstance  $integrationInstance
     */
    public function __construct(
        private readonly IntegrationInstance $integrationInstance,
    ){
        $this->logger = new ApiLogger;
    }

    /**
     * @param string $url
     * @param array|string $body
     * @param string $bodyContentType
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws Exception
     */
    public function post(
        string $url,
        array|string $body,
        string $bodyContentType = 'application/json',
        array $headers = [],
        int $tries = 3,
        ?callable $retryWhen = null
    ): Response{

        return $this->request(
            url: $url,
            method: HttpMethodEnum::POST,
            body: $body,
            bodyContentType: $bodyContentType,
            headers: $headers,
            tries: $tries,
            retryWhen: $retryWhen
        );
    }


    /**
     * @param string $url
     * @param array|string|null $body
     * @param string $bodyContentType
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws IntegrationTimeoutException
     */
    public function delete(
        string $url,
        array|string|null $body = null,
        string $bodyContentType = 'application/json',
        array $headers = [],
        int $tries = 3,
        ?callable $retryWhen = null,
    ): Response{
        return $this->request(
            url: $url,
            method: HttpMethodEnum::DELETE,
            body: $body,
            bodyContentType: $bodyContentType,
            headers: $headers,
            tries: $tries,
            retryWhen: $retryWhen,
        );
    }


    /**
     * @param string $url
     * @param array|string $body
     * @param string $bodyContentType
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws IntegrationTimeoutException
     */
    public function put(
        string $url,
        array|string $body,
        string $bodyContentType = 'application/json',
        array $headers = [],
        int $tries = 3,
        ?callable $retryWhen = null,
    ): Response{
        return $this->request(
            url: $url,
            method: HttpMethodEnum::PUT,
            body: $body,
            bodyContentType: $bodyContentType,
            headers: $headers,
            tries: $tries,
            retryWhen: $retryWhen
        );
    }


    /**
     * @param string $url
     * @param array|string $body
     * @param string $bodyContentType
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws IntegrationTimeoutException
     */
    public function patch(
        string $url,
        array|string $body,
        string $bodyContentType = 'application/json',
        array $headers = [],
        int $tries = 3,
        ?callable $retryWhen = null,
    ): Response{
        return $this->request(
            url: $url,
            method: HttpMethodEnum::PUT,
            body: $body,
            bodyContentType: $bodyContentType,
            headers: $headers,
            tries: $tries,
            retryWhen: $retryWhen
        );
    }


    /**
     * @param string $url
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws Exception
     */
    public function get(
        string $url,
        array $headers = [],
        int $tries = 3,
        ?callable $retryWhen = null
    ): Response{
        return $this->request(
            url: $url,
            method: HttpMethodEnum::GET,
            bodyContentType: '',
            headers: $headers,
            tries: $tries,
            retryWhen: $retryWhen
        );
    }


    /**
     * @param string $url
     * @param HttpMethodEnum $method
     * @param array|string|null $body
     * @param string $bodyContentType
     * @param array $headers
     * @param int $tries
     * @param callable|null $retryWhen
     * @return Response
     * @throws IntegrationTimeoutException
     * @throws Exception
     */
    protected function request(
        string $url,
        HttpMethodEnum $method,
        array|string|null $body = null,
        string $bodyContentType = 'application/json',
        array $headers = [],
        int $tries = 1,
        ?callable $retryWhen = null,
    ): Response{

        if(!is_int($tries) || $tries < 1){
            throw new RuntimeException('Invalid number of tries');
        }

        // Default to retrying when timeout errors occur.
        $retryWhen = $retryWhen ?? function(Throwable $e){
            return $this->isTimeoutError($e);
        };


        return retry($tries, function($attempt) use ($url, $method, $body, $bodyContentType, $headers, $tries){

            $method = strtolower($method->value);

            // Exposing the current log on the object so that
            // it can be retrieved and updated outside of this method.
            $this->currentLogUuid = $this->logger->logApiCallAsync(
                integrationInstance: $this->integrationInstance,
                data: ApiLogData::from([
                    'url' => $url,
                    'requestBody' => is_array($body) ? json_encode($body) : ($body ?? ''),
                    'requestHeaders' => $headers
                ])
            );

            try{

                $request = Http::withHeaders($headers)
                    ->timeout($this->timeout);

                if(!empty($body)){
                    $request = $request->withBody(is_array($body) ? json_encode($body) : $body, $bodyContentType);
                }

                $response = $request->$method($url);

                if(!$response->successful()){

                    // This part is still relevant because some requests may fail
                    // even though they won't throw any request exceptions.

                    // The request failed, we update the API log and throw the exception.
                    dispatch(new UpdateApiLogJob(
                        uuid: $this->currentLogUuid,
                        statusCode: $response->status(),
                        responseHeaders: $response->headers(),
                        responseBody: $response->body()
                    ));

                    if($attempt >= $tries){
                        // The last attempt failed, not as an exception but as an error on the response.
                        // We send the response back to the caller so that it can be handled
                        return $response;
                    }

                    // Throwing an exception will cause the request to be retried
                    // if necessary
                    throw new RuntimeException(
                        'Error running request: ' . $response->reason()
                    );
                } else {
                    dispatch(new UpdateApiLogJob(
                        uuid: $this->currentLogUuid,
                        statusCode: $response->status(),
                        responseHeaders: $response->headers(),
                        responseBody: $response->body()
                    ));

                    return $response;
                }

            }catch (Exception $e){
                // The request failed, we update the API log and throw the exception.
                dispatch(new UpdateApiLogJob(
                    uuid: $this->currentLogUuid,
                    statusCode: $this->isTimeoutError($e) ? 504 : 502,
                    responseHeaders: [],
                    responseBody: $e->getMessage()
                ));

                if($this->isTimeoutError($e) && $attempt >= $tries){
                    // We throw a timeout exception so that it can be handled later.
                    throw new IntegrationTimeoutException($e->getMessage());
                } else {
                    // We throw the original exception so that the request may be retried.
                    throw $e;
                }
            }

        }, 2000, $retryWhen);


    }

    /**
     * @param  Throwable  $e
     * @return bool
     */
    protected function isTimeoutError(Throwable $e): bool{
        return str_contains($e->getMessage(), 'cURL error 28');
    }

    /**
     * @param  int  $timeout
     * @return $this
     */
    public function timeout(int $timeout): self
    {
        if($timeout < 1)
            throw new InvalidArgumentException('Timeout must be a positive integer.');
        $this->timeout = $timeout;
        return $this;
    }

}