<?php

namespace App\Integrations;

use App\Models\Integration;
use Carbon\Carbon;
use DTS\eBaySDK\Constants\SiteIds;
use DTS\eBaySDK\Fulfillment\Services\FulfillmentService;
use DTS\eBaySDK\Inventory\Services\InventoryService;
use DTS\eBaySDK\OAuth\Services\OAuthService;
use DTS\eBaySDK\OAuth\Types\GetUserTokenRestRequest;
use DTS\eBaySDK\OAuth\Types\RefreshUserTokenRestRequest;
use DTS\eBaySDK\Trading\Enums\DetailLevelCodeType;
use DTS\eBaySDK\Trading\Services\TradingService;
use DTS\eBaySDK\Trading\Types\GetOrdersRequestType;
use DTS\eBaySDK\Trading\Types\GetSellerListRequestType;
use DTS\eBaySDK\Trading\Types\PaginationType;
use Exception;
use Generator;
use InvalidArgumentException;

class Ebay extends Channel implements HasOAuth
{
    /**
     * For OAuth authentication.
     *
     * @var OAuthService
     */
    private $oAuthService;

    /**
     * Trading service.
     *
     * @var TradingService
     */
    private $tradingService;

    /**
     * Restful Orders management.
     *
     * @var FulfillmentService
     */
    private $fulfillmentService;

    /**
     * Restful Item management.
     *
     * @var InventoryService
     */
    private $inventoryService;

    /**
     * OAuth user access token.
     *
     * @var string
     */
    private $accessToken;

    public function __construct(array $credentials)
    {
        /**
         * Check required arguments exists in environment.
         */
        if (array_diff(['appId', 'devId', 'certId', 'ruName'], array_keys($credentials))) {
            throw new \InvalidArgumentException('client_id and client_secret are required.');
        }

        /**
         * OAuth service for OAuth function (generate url, get access token...).
         */
        $environment = env('EBAY_ENVIRONMENT', 'sandbox');
        $this->oAuthService = new OAuthService(
            [
                'credentials' => $credentials,
                'ruName' => $credentials['ruName'],
                'sandbox' => $environment == 'sandbox' ? true : false,
            ]
        );

        if (! empty($credentials['access_token'])) {
            /**
             * Trading service.
             */
            $this->tradingService = new TradingService(
                [
                    'credentials' => $credentials,
                    'siteId' => $credentials['siteId'] ?? SiteIds::US,
                    'authToken' => $credentials['access_token'],
                ]
            );

            /**
             * Restful Fulfillment Service to orders management.
             */
            $this->fulfillmentService = new FulfillmentService(
                [
                    'authorization' => $credentials['access_token'],
                ]
            );

            /**
             * Restful Inventory Service to items management.
             */
            $this->inventoryService = new InventoryService(
                [
                    'authorization' => $credentials['access_token'],
                ]
            );

            $this->accessToken = $credentials['access_token'];
        }
    }

    /**
     * Get Application token - access token.
     */
    public function getAppToken(): array
    {
        $api_response = $this->oAuthService->getAppToken();

        return $api_response->toArray();
    }

    /**
     * Get Orders from sales channel will return Generator for pagination.
     *
     * @param  mixed  $options
     */
    public function getSalesOrders($options = null): Generator
    {
        /**
         * Check Access token to ensure trading service.
         */
        if (empty($this->accessToken)) {
            throw new InvalidArgumentException('access_token is required');
        }

        /**
         * New GetOrdersRequestType if $options null or array of options.
         */
        if (empty($options) || is_array($options)) {
            $options = new GetOrdersRequestType($options ?? []);
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options->Pagination)) {
            $initial_page = 1;
            $pagination = new PaginationType();
            $pagination->PageNumber = $initial_page;

            $options->Pagination = $pagination;
        }

        /**
         * fetch orders.
         */
        do {
            $response = $this->tradingService->getOrders($options);

            yield $response;

            $options->Pagination->PageNumber += 1;
            /**
             * Check if has next requests.
             */
            if (isset($response->Errors)) {
                $hasNext = false;
            } elseif ($response->PaginationResult->TotalNumberOfPages > $response->PageNumber) {
                $hasNext = true;
            } else {
                $hasNext = false;
            }
        } while ($hasNext);
        /**
         * If we need to use fulfillmentService.
         */
        //    $request = new GetOrdersRestRequest($options);
        //    return $this->fulfillmentService->getOrders($request);
    }

    public static function handleGetOrdersResponse($response)
    {
    }

    /**
     * Generate OAuth user url.
     */
    public function generateOAuthUrl(array $options = []): array
    {
        $state = uniqid(Integration::NAME_EBAY.'_');
        $url = $this->oAuthService->redirectUrlForUser(
            [
                'state' => $state,
                'scope' => config('sales_channels_oauth.ebay.oauth_scopes'),
            ]
        );

        return ['state' => $state, 'url' => $url];
    }

    /**
     * Get user access token from code.
     *
     *
     * @return mixed
     *
     * @throws Exception
     */
    public function getAccessToken(string $code = '')
    {
        if (empty($code)) {
            throw new Exception('code required.');
        }
        $request = new GetUserTokenRestRequest();
        $request->code = urldecode($code);

        $user_token_response = $this->oAuthService->getUserToken($request);

        if ($user_token_response->getStatusCode() !== 200) {
            throw new Exception($user_token_response->error.': '
                           .$user_token_response->error_description);
        }

        return $user_token_response->toArray();
    }

    /**
     * Refresh user access token.
     *
     *
     * @return mixed
     *
     * @throws Exception
     */
    public function refreshAccessToken(string $refresh_token)
    {
        if (empty($refresh_token)) {
            throw new Exception('refresh_token required.');
        }
        $request = new RefreshUserTokenRestRequest();
        $request->refresh_token = urldecode($refresh_token);
        $request->scope = config('sales_channels_oauth.ebay.oauth_scopes');

        $refresh_response = $this->oAuthService->refreshUserToken($request);

        if ($refresh_response->getStatusCode() !== 200) {
            throw new Exception($refresh_response->error.': '.$refresh_response->error_description);
        }

        return $refresh_response->toArray();
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        // TODO: Implement checkCredentials() method.
    }

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     */
    public function getProducts($options = null): Generator
    {
        /**
         * Check Access token to ensure trading service.
         */
        if (empty($this->accessToken)) {
            throw new InvalidArgumentException('access_token is required');
        }

        /**
         * New GetSellerListRequestType if $options null or array of options.
         */
        if (empty($options) || is_array($options)) {
            $options = new GetSellerListRequestType($options ?? []);
            $options->IncludeVariations = true;
            $options->DetailLevel = [DetailLevelCodeType::C_RETURN_ALL];
            /**
             * To assure we receive Active items.
             */
            $options->EndTimeFrom = Carbon::now()->toDateTime();
            $options->EndTimeTo = Carbon::now()->addDays(120)->toDateTime();
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options->Pagination)) {
            $initial_page = 1;

            $pagination = new PaginationType();
            $pagination->PageNumber = $initial_page;
            $pagination->EntriesPerPage = 50;

            $options->Pagination = $pagination;
        }

        /**
         * fetch orders.
         */
        do {
            $response = $this->tradingService->getSellerList($options);

            yield $response;

            $options->Pagination->PageNumber += 1;
            /**
             * Check if has next requests.
             */
            if (isset($response->Errors)) {
                $hasNext = false;
            } elseif ($response->PaginationResult->TotalNumberOfPages > $response->PageNumber) {
                $hasNext = true;
            } else {
                $hasNext = false;
            }
        } while ($hasNext);
        /**
         * If we need to use inventoryService.
         */
        //    $request = new GetInventoryItemsRestRequest($options ?? []);
        //    return $this->inventoryService->getInventoryItems($request);
    }
}
